<?php

namespace App\Services;

use Throwable;
use Carbon\Carbon;
use RuntimeException;
use App\Models\Saving;
use App\Models\Customer;
use Illuminate\Support\Str;
use App\Models\GeneralLedger;
use App\Models\TargetSavings;

use InvalidArgumentException;
use App\Http\Traites\AuditTraite;

use App\Http\Traites\SavingTraite;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\TargetSavingsProduct;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Models\TargetSavingsSchedule;
use App\Mail\TargetSavingsApprovedMail;
use App\Mail\TargetSavingsTerminalMail;
use App\Models\TargetSavingsTransaction;
use App\Mail\TargetSavingsScheduleFailed;
use App\Mail\TargetSavingsScheduleDebited;
use App\Mail\TargetSavingsInterestCredited;

class TargetSavingsService
{
    use AuditTraite, SavingTraite;

    private const TARGET_SAVINGS_GL_CODE = '40836407';
    private const TARGET_SAVINGS_INTEREST_GL_CODE = '40909777';

    public function createProduct(array $data): TargetSavingsProduct
    {
        $user = Auth::user();
        $data['user_id'] = $user?->id;
        $data['product_code'] = $this->generateUniqueDigits(10);

        return DB::transaction(function () use ($data, $user) {
            $product = TargetSavingsProduct::create($data);
            $this->tracktrails($user?->id, $user?->branch_id, $user?->username, 'Target Savings', "Create Target Savings Product. {$product->id}");
            return $product;
        });
    }

    public function updateProduct(int $id, array $data): TargetSavingsProduct
    {
        $user = Auth::user();
        $data['user_id'] = $user?->id;

        return DB::transaction(function () use ($id, $data, $user) {
            $product = TargetSavingsProduct::findOrFail($id);
            $product->update($data);
            $product->refresh();
            $this->tracktrails($user?->id, $user?->branch_id, $user?->username, 'Target Savings', "Update Target Savings Product. {$product->id}");
            return $product;
        });
    }

    private function generateUniqueDigits(int $length = 10): string
    {
        do {
            $code = $this->randomDigits($length);
        }
        while (TargetSavingsProduct::where('product_code', $code)->exists());
        return $code;
    }

    private function randomDigits(int $length): string
    {
        $s = '';
        for ($i = 0; $i < $length; $i++)
            $s .= (string) random_int(0, 9);
        return $s;
    }

    // ---------------------- HELPERS (Ajax) -----------------------------------
    public function getCustomer($account)
    {
        $customer = Customer::where('acctno', $account)
            ->first(['id', 'user_id', 'branch_id', 'first_name', 'last_name', 'exchangerate_id', 'acctno']);

        if (!$customer)
            return response()->json(['status' => 'error', 'message' => 'Customer not found']);

        $bal = Saving::where('customer_id', $customer->id)->first(['account_balance', 'ledger_balance']);

        $data = [
            'customer_id' => $customer->id,
            'branch_id' => $customer->branch_id,
            'customer_name' => trim($customer->first_name . ' ' . $customer->last_name),
            'acctno' => $customer->acctno,
            'account_balance' => $bal->account_balance ?? '0',
            'ledger_balance' => $bal->ledger_balance ?? '0',
        ];

        return response()->json(['status' => 'success', 'message' => 'Customer retrieved successfully', 'data' => $data]);
    }

    public function getProducteDetails($id)
    {
        $product = TargetSavingsProduct::where('id', $id)
            ->first(['id', 'name', 'product_code', 'minimum_target_deposit', 'interest_rate', 'minimum_hold_days', 'description']);

        if (!$product)
            return response()->json(['status' => 'error', 'message' => 'Product not found']);

        return response()->json(['status' => 'success', 'message' => 'Product retrieved successfully', 'data' => $product]);
    }

    // ---------------------- CREATE TARGETS -----------------------------------
    public function storeAutoSaveTargetSavings(array $validated): TargetSavings
    {
        $productId = (int) ($validated['target_savings_products_id'] ?? 0);
        $customerId = (int) $validated['customer_id'];

        $product = TargetSavingsProduct::find($productId);
        $customer = Customer::find($customerId);
        $user = Auth::user();

        if (!$product)
            throw new InvalidArgumentException('Invalid Product ID');
        if (!$customer)
            throw new InvalidArgumentException('Invalid Customer');

        $minTarget = (float) ($product->minimum_deposit ?? $product->minimum_target_deposit ?? 0);
        if ((float) $validated['target_amount'] < $minTarget) {
            throw new InvalidArgumentException('Target amount must be at least ' . number_format($minTarget, 2));
        }

        $savings = Saving::where('customer_id', $customerId)->first();
        $availableBalance = (float) ($savings->account_balance ?? 0);
        if ((float) $validated['auto_save_amount'] > $availableBalance) {
            throw new InvalidArgumentException('Insufficient account balance');
        }

        $code = $this->uniqueTargetCode(10);

        return DB::transaction(function () use ($validated, $product, $user, $code, $productId) {
            $data = [
                'code' => $code,
                'customer_id' => (int) $validated['customer_id'],
                'user_id' => $user?->id,
                'target_savings_product_id' => (int) $productId,
                'account_number' => $validated['account_number'],
                'savings_plan_name' => $validated['savings_plan_name'],
                'target_amount' => (float) $validated['target_amount'],
                'duration' => (int) ($product->minimum_hold_days ?? 0),
                'auto_save' => true,
                'frequency' => $validated['frequency'],
                'auto_save_start_date' => $validated['auto_save_start_date'],
                'auto_save_amount' => (float) ($validated['auto_save_amount'] ?? 0),
                'status' => 'pending',
                'start_date' => null,
                'maturity_date' => null,
                'interest_method' => 'daily',
                'created_by' => (string) ($user?->id ?? 'SYSTEM'),
                'created_date' => Carbon::now()->toDateTimeString(),
                'first_successful_auto_debit_at' => null,
                'last_interest_accrued_at' => null,
            ];

            $target = TargetSavings::create($data);
            $this->tracktrails($user?->id, $user?->branch_id, $user?->username, 'Target Savings', "Create Target Savings. {$target->code}");
            return $target;
        });
    }

    public function storeTargetsavings(array $validated): TargetSavings
    {
        $productId = (int) ($validated['target_savings_products_id'] ?? 0);
        $customerId = (int) $validated['customer_id'];

        $product = TargetSavingsProduct::find($productId);
        $customer = Customer::find($customerId);
        $user = Auth::user();

        if (!$product)
            throw new InvalidArgumentException('Invalid Product ID');
        if (!$customer)
            throw new InvalidArgumentException('Invalid Customer');

        $minTarget = (float) ($product->minimum_deposit ?? $product->minimum_target_deposit ?? 0);
        if ((float) $validated['target_amount'] < $minTarget) {
            throw new InvalidArgumentException('Target amount must be at least ' . number_format($minTarget, 2));
        }

        $code = $this->uniqueTargetCode(10);

        return DB::transaction(function () use ($validated, $product, $user, $code, $productId) {
            $data = [
                'code' => $code,
                'customer_id' => (int) $validated['customer_id'],
                'user_id' => $user?->id,
                'target_savings_product_id' => (int) $productId,
                'account_number' => $validated['account_number'],
                'savings_plan_name' => $validated['savings_plan_name'],
                'target_amount' => (float) $validated['target_amount'],
                'duration' => (int) ($product->minimum_hold_days ?? 0),
                'auto_save' => false,
                'frequency' => null,
                'auto_save_start_date' => null,
                'auto_save_amount' => 0.00,
                'status' => 'active',
                'start_date' => null,
                'maturity_date' => null,
                'interest_method' => 'daily',
                'created_by' => (string) ($user?->id ?? 'SYSTEM'),
                'created_date' => Carbon::now()->toDateTimeString(),
                'first_successful_auto_debit_at' => null,
                'last_interest_accrued_at' => null,
            ];

            $target = TargetSavings::create($data);
            $this->tracktrails($user?->id, $user?->branch_id, $user?->username, 'Target Savings', "Create Target Savings. {$target->code}");
            return $target;
        });
    }

    private function uniqueTargetCode(int $length = 10): string
    {
        do {
            $code = 'TS' . $this->randomDigits($length);
        }
        while (TargetSavings::where('code', $code)->exists());
        return $code;
    }

    // ---------------------- UPDATE TARGET ------------------------------------
    public function updateTargetsavings(int $id, array $validated): TargetSavings
    {
        $target = TargetSavings::findOrFail($id);
        $productId = (int) ($validated['target_savings_products_id'] ?? 0);
        $product = TargetSavingsProduct::findOrFail($productId);

        if (!in_array($target->status, ['pending'])) {
            throw new InvalidArgumentException('Only pending target savings can be edited.');
        }

        $minTarget = (float) ($product->minimum_deposit ?? $product->minimum_target_deposit ?? 0);
        if ((float) $validated['target_amount'] < $minTarget) {
            throw new InvalidArgumentException('Target amount must be at least ' . number_format($minTarget, 2));
        }

        $saving = Saving::where('customer_id', (int) $validated['customer_id'])->first();
        $balance = (float) ($saving->account_balance ?? 0);
        if ((float) $validated['auto_save_amount'] > $balance) {
            throw new InvalidArgumentException('Insufficient account balance for the specified Save Now amount.');
        }

        $autoDate = Carbon::parse($validated['auto_save_start_date']);
        $today = Carbon::today();
        if ($autoDate->lt($today) && in_array($target->status, ['pending'])) {
            $autoDate = $today;
        }

        $user = Auth::user();

        return DB::transaction(function () use ($target, $product, $validated, $autoDate, $user, $productId) {
            $payload = [
                'target_savings_product_id' => (int) $productId,
                'customer_id' => (int) $validated['customer_id'],
                'account_number' => $validated['account_number'],
                'savings_plan_name' => $validated['savings_plan_name'],
                'target_amount' => (float) $validated['target_amount'],
                'frequency' => $validated['frequency'],
                'auto_save' => (bool) ($validated['auto_save'] ?? false),
                'auto_save_start_date' => $autoDate->toDateString(),
                'auto_save_amount' => (float) $validated['auto_save_amount'],
                'duration' => (int) ($product->minimum_hold_days ?? 0),
                'updated_by' => (string) ($user?->id ?? 'SYSTEM'),
                'updated_date' => Carbon::now()->toDateTimeString(),
            ];

            $target->update($payload);
            $target->refresh();

            $this->tracktrails($user?->id, $user?->branch_id, $user?->username, 'Target Savings', "Update Target Savings. {$target->id}");
            return $target;
        });
    }


    public function approveAndDebitTargetSavings(int $id)
    {
        /** @var TargetSavings $target */
        $target = TargetSavings::findOrFail($id);

        if ($target->status !== 'pending') {
            return back()->with('error', 'Only pending Target Savings can be approved.');
        }

        if ($target->user_id === Auth::id()) {
            return back()->with('error', 'You cannot approve a Target Savings  that you booked.');
        }



        $today = Carbon::today();
        $startDate = $target->auto_save_start_date
            ? Carbon::parse($target->auto_save_start_date)->startOfDay()
            : $today->copy()->startOfDay();

        // Work out schedule counts and per-installment amount
        $perDebit = $target->auto_save_amount ? (float) $target->auto_save_amount : null;
        if ($perDebit && $perDebit > 0) {
            $installments = (int) ceil(((float) $target->target_amount) / $perDebit);
            $amountEach = $perDebit;
        } else {
            [$installments, $amountEach] = $this->deriveFromDuration($target);
        }

        $actor = auth()->user();
        $actorId = (int) ($actor->id ?? 0);
        $actorTag = (string) ($actor->username ?? $actorId ?: 'SYSTEM');
        $branchId = $actor->branch_id ?? null;
        $device = request()->userAgent() ?? 'core';

        // === 1) APPROVE + BUILD SCHEDULES ======================================
        DB::transaction(function () use ($target, $startDate, $installments, $amountEach, $actorId, $actorTag) {
            // Clear any existing schedules for idempotency
            TargetSavingsSchedule::where('target_savings_id', $target->id)->delete();

            $due = $startDate->copy();
            $sum = 0.0;
            $lastDue = $startDate->copy();

            for ($i = 1; $i <= $installments; $i++) {
                $amount = ($i < $installments)
                    ? round($amountEach, 2)
                    : round(((float) $target->target_amount - $sum), 2);

                TargetSavingsSchedule::create([
                    'target_savings_id' => $target->id,
                    'target_savings_code' => $target->code,
                    'schedule_no' => $i,
                    'due_date' => $due->toDateString(),
                    'due_at' => $due->copy()->startOfDay(),
                    'amount' => $amount,
                    'status' => 'pending',
                    'attempt_count' => 0,
                    'paid_at' => null,
                ]);

                $sum += $amount;
                $lastDue = $due->copy();
                $due = $this->advance($due, $target->frequency);
            }

            // Mark approved + key dates
            $target->status = 'approved';
            $target->approved_by = $actorTag;
            $target->approved_at = now();
            $target->start_date = $target->start_date ?: $startDate->toDateString();
            $target->maturity_date = $target->maturity_date ?: $lastDue->toDateString();
            $target->save();
        });

        $this->tracktrails($actorId, $branchId, $actorTag, 'Target Savings', "Approve & build schedule. {$target->code}");

        // === 2) IMMEDIATE FIRST DEBIT (and records) ============================
        // We always attempt to debit schedule #1 right now. If funds are insufficient,
        // we keep the approval & schedules and just return a clear error.
        $first = TargetSavingsSchedule::where('target_savings_id', $target->id)
            ->where('status', 'pending')
            ->orderBy('schedule_no', 'asc')
            ->first();

        if (!$first) {
            return back()->with('error', 'Approved and schedules created, but no pending schedule found to debit.');
        }

        try {
            DB::transaction(function () use ($target, $first, $actorId, $actorTag, $branchId, $device) {
                $savings = Saving::where('customer_id', $target->customer_id)->lockForUpdate()->first();
                if (!$savings) {
                    $first->attempt_count = ($first->attempt_count ?? 0) + 1;
                    $first->save();
                    throw new RuntimeException('Savings account not found for customer.');
                }

                $poolGL = GeneralLedger::where('gl_code', self::TARGET_SAVINGS_GL_CODE)->lockForUpdate()->firstOrFail();

                $bal = (float) $savings->account_balance;
                $ledger = (float) $savings->ledger_balance;
                $lien = (float) ($savings->lien_amount ?? 0);
                $amount = (float) $first->amount;
                $avail = $bal - $lien;

                if ($amount <= 0) {
                    $first->attempt_count = ($first->attempt_count ?? 0) + 1;
                    $first->save();
                    throw new RuntimeException('Invalid debit amount.');
                }
                if ($avail < $amount) {
                    $first->attempt_count = ($first->attempt_count ?? 0) + 1;
                    $first->save();
                    throw new RuntimeException('Insufficient funds in savings.');
                }

                $ref = 'TXN' . Str::ulid();

                // 2a) Debit customer savings
                $savings->account_balance = $this->numToDb(round($bal - $amount, 2));
                $savings->ledger_balance = $this->numToDb(round($ledger - $amount, 2));
                $savings->save();

                $this->create_saving_transaction(
                    $actorId,
                    $target->customer_id,
                    $branchId,
                    $amount,
                    'withdrawal',
                    $device,
                    0,
                    null,
                    1,
                    'internal',
                    $savings->account_number ?? $target->account_number,
                    $ref,
                    'Target savings funding ' . $target->code . ' (schedule #1)',
                    'success',
                    'debit',
                    'system',
                    $actorTag
                );

                // 2b) Increase Target Savings liability pool (GL)
                $this->gltransaction('withdrawal', $poolGL, $amount, $branchId);

                $this->create_saving_transaction_gl(
                    $actorId,
                    $poolGL->id,
                    $branchId,
                    $amount,
                    'deposit',
                    $device,
                    null,
                    $ref,
                    'Target savings funding ' . $target->code . ' (schedule #1)',
                    'success',
                    $actorTag
                );

                // 2c) Mark schedule paid
                $first->status = 'paid';
                $first->paid_at = now();
                $first->attempt_count = ($first->attempt_count ?? 0) + 1;
                $first->save();

                // 2d) Update target running totals
                $target->total_amount_saved = $this->numToDb(
                    round(((float) $target->total_amount_saved) + $amount, 2)
                );
                $target->save();

                // 2e) Record TargetSavingsTransaction
                $this->createTargetSavingTransactions(
                    $target->id,
                    $first->id,
                    $target->customer_id,
                    $actorId,
                    $target->code,
                    'credit',
                    $amount,
                    $device,
                    $ref,
                    'Initial debit after approval (schedule #1)',
                    null,
                    'success',
                    $actorId,
                    $actorId,
                    $actorTag
                );

                // 2f) Mark first successful debit date (starts daily interest)
                $this->markFirstDebitIfNeeded($target, Carbon::parse($first->paid_at));


                $customer = Customer::select('first_name', 'last_name', 'email')->find($target->customer_id);
                if ($customer && $customer->email) {

                    $customerName = trim(($customer->first_name ?? '') . ' ' . ($customer->last_name ?? ''));

                    $data = [
                        'type' => 'Target Savings Approved',
                        'preheader' => 'Your Target Savings has been approved.',
                        'customer_name' => $customerName,
                        'code' => $target->code,
                        'plan_name' => $target->savings_plan_name,
                        'start_date' => $target->start_date,
                        'maturity_date' => $target->maturity_date,
                        'amount' => number_format((float) $target->auto_save_amount, 2),
                        'frequency' => $target->frequency,
                    ];

                    Mail::to($customer->email)->queue(new TargetSavingsApprovedMail($data));

                }



                $this->tracktrails(
                    $actorId,
                    $branchId,
                    $actorTag,
                    'Target Savings',
                    "Immediate debit success. Target {$target->code}, Schedule {$first->id}, Amount {$amount}"
                );
            });

            return back()->with('success', 'Approved, schedules created, and first debit completed.');

        } catch (Throwable $e) {
            Log::warning('approveAndDebitTargetSavings: debit failed', [
                'target_id' => $target->id,
                'schedule_id' => $first?->id,
                'error' => $e->getMessage(),
            ]);

            // Approval stands; schedules exist. Tell user why debit failed.
            return back()->with(
                'error',
                'Approved and schedules created, but first debit failed: ' . $e->getMessage()
            );
        }
    }


    private function attemptFirstAutoSave($target): array
    {
        $first = TargetSavingsSchedule::where('target_savings_id', $target->id)
            ->where('status', 'pending')
            ->orderBy('schedule_no', 'asc')
            ->first();

        if (!$first)
            return ['ok' => false, 'message' => 'No pending schedule found to auto-debit.'];

        try {
            DB::transaction(function () use ($target, $first) {
                $actor = auth()->user();
                $actorId = (int) ($actor->id ?? 0);
                $actorTag = (string) ($actor->username ?? $actorId ?: 'SYSTEM');
                $branchId = $actor->branch_id ?? null;
                $device = request()->userAgent() ?? 'core';

                $savings = Saving::where('customer_id', $target->customer_id)->lockForUpdate()->first();
                if (!$savings)
                    throw new RuntimeException('Savings account not found for customer.');

                $poolGL = GeneralLedger::where('gl_code', self::TARGET_SAVINGS_GL_CODE)->lockForUpdate()->firstOrFail();

                $bal = (float) $savings->account_balance;
                $ledger = (float) $savings->ledger_balance;
                $lien = (float) ($savings->lien_amount ?? 0);
                $amount = (float) $first->amount;
                $avail = $bal - $lien;

                if ($amount <= 0)
                    throw new RuntimeException('Invalid debit amount.');
                if ($avail < $amount) {
                    $first->attempt_count = ($first->attempt_count ?? 0) + 1;
                    $first->save();
                    throw new RuntimeException('Insufficient funds in savings.');
                }

                $ref = 'TXN' . Str::ulid();

                $savings->account_balance = $this->numToDb(round($bal - $amount, 2));
                $savings->ledger_balance = $this->numToDb(round($ledger - $amount, 2));
                $savings->save();

                $this->create_saving_transaction(
                    $actorId,
                    $target->customer_id,
                    $branchId,
                    $amount,
                    'withdrawal',
                    $device,
                    0,
                    null,
                    1,
                    'internal',
                    $savings->account_number ?? $target->account_number,
                    $ref,
                    'Auto-save to target ' . $target->code,
                    'success',
                    'debit',
                    'system',
                    $actorTag
                );

                // Increase liability pool
                $this->gltransaction('withdrawal', $poolGL, $amount, $branchId);

                $this->create_saving_transaction_gl(
                    $actorId,
                    $poolGL->id,
                    $branchId,
                    $amount,
                    'deposit',
                    $device,
                    null,
                    $ref,
                    'Auto-save funding ' . $target->code,
                    'success',
                    $actorTag
                );

                $first->status = 'paid';
                $first->paid_at = now();
                $first->attempt_count = ($first->attempt_count ?? 0) + 1;
                $first->save();

                $target->total_amount_saved = $this->numToDb(
                    round(((float) $target->total_amount_saved) + $amount, 2)
                );
                $target->save();

                $this->createTargetSavingTransactions(
                    $target->id,
                    $first->id,
                    $target->customer_id,
                    $actorId,
                    $target->code,
                    'credit',
                    $amount,
                    $device,
                    $ref,
                    'Auto-save debit (first schedule)',
                    null,
                    'success',
                    $actorId,
                    $actorId,
                    $actorTag
                );

                // Mark first successful debit date (enables interest start)
                $this->markFirstDebitIfNeeded($target, Carbon::parse($first->paid_at));

                $this->tracktrails(
                    $actorId,
                    $branchId,
                    $actorTag,
                    'Target Savings',
                    "Auto-debit first schedule. Target {$target->id}, Schedule {$first->id}, Amount {$amount}"
                );
            });

            return ['ok' => true, 'message' => 'Schedules generated and first auto-save debit completed for target: ' . $target->code];

        } catch (Throwable $e) {
            // Whole TX rolled back automatically
            Log::warning('First auto-save debit failed', ['target_id' => $target->id, 'error' => $e->getMessage()]);
            return ['ok' => false, 'message' => 'Schedules created, but first auto-save debit failed: ' . $e->getMessage()];
        }
    }

    // ---------------------- SCHEDULE CRON (AUTO-DEBIT) -----------------------
    public function ScheduleCronJob(int $scheduleId): array
    {
        $sch = TargetSavingsSchedule::findOrFail($scheduleId);
        if ($sch->status !== 'pending')
            return ['ok' => false, 'message' => "Skipped: status '{$sch->status}' is not pending"];

        $target = TargetSavings::lockForUpdate()->findOrFail($sch->target_savings_id);
        $customer = Customer::find($target->customer_id);

        try {
            DB::transaction(function () use ($sch, $target, $customer) {
                $savings = Saving::where('customer_id', $target->customer_id)->lockForUpdate()->first();
                if (!$savings)
                    throw new RuntimeException('Savings account not found.');

                $sch->refresh();
                if ($sch->status !== 'pending')
                    return;

                $amount = (float) $sch->amount;
                if ($amount <= 0)
                    throw new RuntimeException('Invalid debit amount.');

                $bal = (float) $savings->account_balance;
                $ledger = (float) $savings->ledger_balance;
                $lien = (float) ($savings->lien_amount ?? 0);
                $avail = $bal - $lien;

                $device = 'Core';
                $branchId = $customer->branch_id ?? null;
                $authTag = 'SYSTEM';

                if ($avail < $amount) {
                    $sch->attempt_count = ($sch->attempt_count ?? 0) + 1;
                    $sch->updated_by = $authTag;
                    $sch->save();

                    if ($customer && $customer->email) {
                        Mail::to($customer->email)->queue(
                            new TargetSavingsScheduleFailed($target, $sch, 'Insufficient funds')
                        );
                    }
                    throw new RuntimeException('Insufficient funds.');
                }

                $savings->account_balance = $this->numToDb(round($bal - $amount, 2));
                $savings->ledger_balance = $this->numToDb(round($ledger - $amount, 2));
                $savings->save();

                $sch->status = 'paid';
                $sch->paid_at = now();
                $sch->attempt_count = ($sch->attempt_count ?? 0) + 1;
                $sch->updated_by = $authTag;
                $sch->save();

                $target->total_amount_saved = $this->numToDb(
                    round(((float) $target->total_amount_saved) + $amount, 2)
                );
                $target->updated_by = $authTag;
                $target->save();

                $ref = 'TXN' . Str::uuid();
                $poolGL = GeneralLedger::where('gl_code', self::TARGET_SAVINGS_GL_CODE)->lockForUpdate()->firstOrFail();

                $this->create_saving_transaction(
                    null,
                    $target->customer_id,
                    $branchId,
                    $amount,
                    'withdrawal',
                    $device,
                    0,
                    null,
                    1,
                    'internal',
                    $target->account_number,
                    $ref,
                    'Auto-debit to target ' . $target->code . ' (schedule #' . $sch->schedule_no . ')',
                    'success',
                    'debit',
                    'system',
                    $authTag
                );

                // Increase liability pool
                $this->gltransaction('withdrawal', $poolGL, $amount, $branchId);

                $this->create_saving_transaction_gl(
                    null,
                    $poolGL->id,
                    $branchId,
                    $amount,
                    'deposit',
                    $device,
                    null,
                    $ref,
                    'Auto-save funding ' . $target->code . ' (schedule #' . $sch->schedule_no . ')',
                    'success',
                    $authTag
                );

                $this->createTargetSavingTransactions(
                    $target->id,
                    $sch->id,
                    $target->customer_id,
                    null,
                    $target->code,
                    'credit',
                    $amount,
                    $device,
                    'Auto-debit for schedule #' . $sch->schedule_no,
                    null,
                    'success',
                    $authTag,
                    'Core'
                );

                if ($customer && $customer->email) {
                    Mail::to($customer->email)->queue(
                        new TargetSavingsScheduleDebited($target, $sch, $amount)
                    );
                }

                $this->tracktrails(
                    null,
                    $branchId,
                    null,
                    'Target Savings',
                    "Auto-debit SUCCESS for target {$target->target}, schedule #{$sch->id}"
                );

                // Mark first successful debit date (if not set)
                $this->markFirstDebitIfNeeded($target, Carbon::parse($sch->paid_at));
            });

            return ['status' => 'success', 'status_code' => '00', 'message' => 'Debited and recorded'];

        } catch (Throwable $e) {
            // rollback has already occurred
            Log::warning('ScheduleCronJob failed', [
                'schedule_id' => $scheduleId,
                'error' => $e->getMessage(),
            ]);

            $code = str_contains(strtolower($e->getMessage()), 'insufficient') ? '51' : '96';
            return ['status' => 'failed', 'status_code' => $code, 'message' => $e->getMessage()];
        }
    }

    // ---------------------- REJECT / TERMINATE / DISBURSE --------------------
    public function rejectTargetsavings($id)
    {
        $target = TargetSavings::where('id', $id)->firstOrFail();
        if ($target->status === 'active') {
            return back()->with('error', "You cannot reject an $target->status Target Savings.");
        }

        DB::transaction(function () use ($target) {
            $target->update([
                'status' => 'rejected',
                'declined_by' => auth()->id(),
                'declined_at' => now(),
            ]);
        });

        return back()->with('success', 'Target savings rejected successfully.');
    }

    public function terminateTargetSavings($id)
    {
        $target = TargetSavings::where('id', $id)->firstOrFail();
        if (in_array($target->status, ['closed', 'matured', 'terminated'])) {
            return back()->with('error', 'Target Savings is already closed, matured, or terminated.');
        }

        $auth = Auth::user();
        $customer = Customer::findOrFail($target->customer_id);

        DB::transaction(function () use ($target) {
            TargetSavingsSchedule::where('target_savings_id', $target->id)
                ->where('target_savings_code', $target->code)
                ->update([
                    'status' => 'terminated',
                    'terminated_by' => auth()->id(),
                    'terminated_at' => now(),
                    'updated_by' => auth()->id(),
                    'updated_at' => now(),
                ]);

            $target->update([
                'status' => 'terminated',
                'terminated_by' => auth()->id(),
                'terminated_at' => now(),
                'updated_by' => auth()->id(),
            ]);
        });

        Mail::to($customer->email)->queue(new TargetSavingsTerminalMail([
            'savings_plan_name' => $target->savings_plan_name,
            'code' => $target->code,
        ]));

        $this->tracktrails($auth->id, $auth->branch_id, $auth->username, 'Target Savings', "Terminate Target Savings. {$target->code}");
        return back()->with('success', 'Target Savings terminated successfully.');
    }


    // -- INTEREST ACCRUAL (DAILY) --
    public function accrueDailyInterestCron(): array
    {
        $today = Carbon::today();

        // System (headless cron) context
        $ctx = $this->systemContext();
        $actorId = $ctx['actorId'];   // null
        $actorTag = $ctx['actorTag'];  // 'SYSTEM'
        $branchId = $ctx['branchId'];  // 'SYSTEM'

        $targets = TargetSavings::query()
            ->where('status', 'active')
            ->whereNotNull('first_successful_auto_debit_at')
            ->get();

        $processed = 0;
        $skipped = 0;

        foreach ($targets as $t) {
            try {
                DB::transaction(function () use ($t, $today, $actorId, $actorTag, $branchId) {
                    /** @var TargetSavings $target */
                    $target = TargetSavings::lockForUpdate()->find($t->id);
                    if (!$target) {
                        return;
                    }

                    // Idempotency by calendar day
                    $lastAccrued = $target->last_interest_accrued_at
                        ? Carbon::parse($target->last_interest_accrued_at)
                        : null;

                    $startAccrualFrom = $lastAccrued
                        ? $lastAccrued->copy()->addDay()
                        : Carbon::parse($target->first_successful_auto_debit_at)->startOfDay();

                    if ($startAccrualFrom->gt($today)) {
                        // AUDIT: skip (already up to date)
                        $this->tracktrails(
                            $actorId,
                            $branchId,
                            $actorTag,
                            'Target Savings',
                            "Skip interest accrual: up-to-date for {$target->code} (last={$target->last_interest_accrued_at})"
                        );
                        return;
                    }

                    // Determine daily rate from product
                    $product = TargetSavingsProduct::findOrFail($target->target_savings_product_id);
                    $dailyRate = $this->getDailyRate((float) $product->interest_rate); // e.g. 12 => 0.12/365

                    // Build daily payments map for the span (so we accrue on end-of-day balance)
                    $byDayPaid = TargetSavingsSchedule::query()
                        ->where('target_savings_id', $target->id)
                        ->where('status', 'paid')
                        ->whereDate('paid_at', '>=', $startAccrualFrom->toDateString())
                        ->whereDate('paid_at', '<=', $today->toDateString())
                        ->get()
                        ->groupBy(fn($s) => Carbon::parse($s->paid_at)->toDateString())
                        ->map(fn($g) => (float) $g->sum('amount'));

                    // Reconstruct opening balance at day-1
                    $latestBalance = (float) ($target->total_amount_saved ?? 0);
                    $rollbackSum = (float) array_sum($byDayPaid->all());
                    $opening = max(0.0, $latestBalance - $rollbackSum);

                    $cursor = $startAccrualFrom->copy();
                    while ($cursor->lte($today)) {
                        $dateKey = $cursor->toDateString();
                        $paidToday = (float) ($byDayPaid[$dateKey] ?? 0.0);

                        // End-of-day balance = opening + today's paid schedules
                        $opening += $paidToday;
                        $eodBalance = $opening;

                        // Compute interest for the day
                        $interestToday = $this->safeRound($eodBalance * $dailyRate, 2);

                        if ($interestToday > 0) {
                            // Book GL: DR Interest Expense, CR Target Savings Liability
                            $expenseGL = GeneralLedger::where('gl_code', self::TARGET_SAVINGS_INTEREST_GL_CODE)->lockForUpdate()->firstOrFail();
                            $poolGL = GeneralLedger::where('gl_code', self::TARGET_SAVINGS_GL_CODE)->lockForUpdate()->firstOrFail();
                            $device = 'SYSTEM';
                            $ref = 'INT' . Str::ulid();

                            // DR Expense (increase)
                            $this->gltransaction('deposit', $expenseGL, $interestToday, $branchId);
                            // CR Liability (increase)
                            $this->gltransaction('withdrawal', $poolGL, $interestToday, $branchId);

                            // Record TargetSavingsTransaction (credit to target)
                            $this->createTargetSavingTransactions(
                                $target->id,
                                null,
                                $target->customer_id,
                                null,
                                $target->code,
                                'credit',
                                $interestToday,
                                $device,
                                $ref,
                                'Daily interest accrual (' . $dateKey . ')',
                                null,
                                'success',
                                'SYSTEM',
                                'SYSTEM',
                                null
                            );

                            // Update totals
                            $target->total_accrued_interest = $this->numToDb(
                                $this->safeRound(((float) $target->total_accrued_interest) + $interestToday, 2)
                            );
                            $target->save();

                            // AUDIT: interest accrued for the day
                            $this->tracktrails(
                                $actorId,
                                $branchId,
                                $actorTag,
                                'Target Savings',
                                "Accrued daily interest for {$target->code} on {$dateKey}: amt=" . number_format($interestToday, 2) .
                                ", eod_bal=" . number_format($eodBalance, 2) .
                                ", annual_rate=" . ($product->interest_rate ?? 0) . "%, daily_rate=" . number_format($dailyRate, 8) .
                                ", gl_ref={$ref}"
                            );
                        } else {
                            // AUDIT: zero-interest day (e.g., zero balance)
                            $this->tracktrails(
                                $actorId,
                                $branchId,
                                $actorTag,
                                'Target Savings',
                                "Zero-interest day for {$target->code} on {$dateKey}: eod_bal=" . number_format($eodBalance, 2) .
                                ", daily_rate=" . number_format($dailyRate, 8)
                            );
                        }

                        // Mark day accrued (idempotency)
                        $target->last_interest_accrued_at = $dateKey;
                        $target->save();

                        // Prepare and send email (minimal payload)
                        $customer = Customer::select('first_name', 'last_name', 'email')->find($target->customer_id);
                        if ($customer && $customer->email) {
                            $customerName = trim($customer->first_name . ' ' . $customer->last_name);
                            $data = [
                                'customer_name' => $customerName,
                                'interest' => number_format((float) $interestToday, 2),
                                'date_key' => $dateKey,
                            ];

                            Mail::to($customer->email)->queue(
                                new TargetSavingsInterestCredited($data)
                            );

                            // AUDIT: email queued
                            $this->tracktrails(
                                $actorId,
                                $branchId,
                                $actorTag,
                                'Target Savings',
                                "Interest email queued for {$target->code} on {$dateKey} to {$customer->email} (interest=" .
                                number_format((float) $interestToday, 2) . ")"
                            );
                        }

                        $cursor->addDay();
                    }
                });

                $processed++;
            } catch (Throwable $e) {
                // Any failure rolls back the target’s accrual for that run
                Log::warning('accrueDailyInterestCron skipped target', [
                    'target_id' => $t->id,
                    'error' => $e->getMessage(),
                ]);

                // AUDIT: failure for this target
                $this->tracktrails(
                    $actorId,
                    $branchId,
                    $actorTag,
                    'Target Savings',
                    "Interest accrual FAILED for target {$t->id}: " . $e->getMessage()
                );

                $skipped++;
            }
        }

        return [
            'status' => 'success',
            'message' => 'ok',
            'processed' => $processed,
            'skipped' => $skipped,
        ];
    }


    // ---------------------- SHARED HELPERS -----------------------------------
    private function markFirstDebitIfNeeded(TargetSavings $target, Carbon $paidAt): void
    {
        if (empty($target->first_successful_auto_debit_at)) {
            $target->first_successful_auto_debit_at = $paidAt->toDateString();
            $target->last_interest_accrued_at = null; // start fresh
            $target->save();
        }
    }

    private function deriveFromDuration($t): array
    {
        $periods = (int) ($t->duration ?? 0);
        if ($periods <= 0)
            $periods = 1;
        $amountEach = floor(((float) $t->target_amount / $periods) * 100) / 100;
        return [$periods, $amountEach];
    }

    private function advance(Carbon $date, string $frequency): Carbon
    {
        return match ($frequency) {
            'daily' => $date->copy()->addDay(),
            'weekly' => $date->copy()->addWeek(),
            'monthly' => $date->copy()->addMonth(),
            default => $date->copy()->addDay(),
        };
    }

    private function getDailyRate(float $annualPercent): float
    {
        return ($annualPercent / 100.0) / 365.0;
    }

    private function safeRound(float $n, int $dp = 2): float
    {
        return (float) number_format($n, $dp, '.', '');
    }

    private function numToDb(float $n): string
    {
        return number_format($n, 2, '.', '');
    }

    // ---------------------- TXN RECORD WRITER --------------------------------
    public function createTargetSavingTransactions(
        $targetSavinsId,
        $targetScheduleId,
        $customerId,
        $userId,
        $targetCode,
        $type,
        $amount,
        $device,
        $ref,
        $narration,
        $note,
        $status,
        $createBy,
        $initiatedBy = null,
        $disbursedBy = null
    ) {
        TargetSavingsTransaction::create([
            'target_savings_id' => $targetSavinsId,
            'target_savings_schedule_id' => $targetScheduleId,
            'customer_id' => $customerId,
            'user_id' => $userId,
            'target_savings_code' => $targetCode,
            'type' => $type,
            'amount' => $amount,
            'device' => $device,
            'reference_no' => is_string($ref) ? $ref : (string) $ref,
            'narration' => $narration,
            'note' => $note,
            'status' => $status,
            'created_by' => $createBy,
            'initiated_by' => $initiatedBy,
            'initiated_at' => now(),
            'disbursed_by' => $disbursedBy,
            'disbursed_at' => $disbursedBy ? now() : null,
        ]);
    }

    private function systemContext(): array
    {
        return [
            'actorId' => 0,
            'actorTag' => 'SYSTEM',
            'branchId' => 'SYSTEM',
        ];
    }

}
