<?php

namespace App\Services;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\Customer;
use App\Models\Saving;
use App\Models\GeneralLedger;
use App\Models\MonthlySavingsProfit;
use App\Models\Setting;
use App\Http\Traites\AuditTraite;
use App\Http\Traites\SavingTraite;

class SettingsService
{
    use AuditTraite, SavingTraite;

    // ONE GL ONLY — Interest Expense GL to be debited when paying profit
    private const INTEREST_EXPENSE_GL = '20311399';

    public function applyMonthlySavingsProfit(): array
    {
        $settings = new Setting();

        // Settings
        $interestRate    = (float) ($settings->getsettingskey('savings_account_interest_rate') ?? 0);   // e.g. 2.5
        $withdrawalLimit = (int)   ($settings->getsettingskey('savings_account_transfer_limit_per_month') ?? 0); // e.g. 3 (0 = no limit)

        // Rates
        $annualRate  = $interestRate / 100;
        $monthlyRate = $annualRate / 12;

        $actor = 'SYSTEM';
        $userId = null;
        $tz = 'Africa/Lagos';

        $now        = Carbon::now($tz);
        $monthLabel = $now->format('Y-m'); // e.g. '2025-11'
        $monthStart = $now->copy()->startOfMonth()->startOfDay();
        $monthEnd   = $now->copy()->endOfMonth()->endOfDay();

        $countsSub = DB::table('savings_transactions as st')
            ->select('st.customer_id', DB::raw('COUNT(*) as withdrawals_total'))
            ->whereRaw('LOWER(st.type) = ?', ['withdrawal'])
            ->whereBetween('st.created_at', [$monthStart, $monthEnd])
            ->groupBy('st.customer_id');

        $base = Customer::with('savings')
            ->leftJoinSub($countsSub, 'wc', function ($join) {
                $join->on('wc.customer_id', '=', 'customers.id');
            })
            ->select('customers.*', DB::raw('COALESCE(wc.withdrawals_total, 0) as withdrawals_total'));

        if ($withdrawalLimit > 0) {
            $base->whereRaw('COALESCE(wc.withdrawals_total, 0) <= ?', [$withdrawalLimit]);
        }

        $customers = $base->get();

        $payload = [];

        foreach ($customers as $customer) {
            DB::transaction(function () use ($customer, $interestRate, $monthlyRate, $monthLabel, $actor, $userId, &$payload) {
                // Idempotency: already paid for this month?
                $alreadyPaid = MonthlySavingsProfit::where('customer_id', $customer->id)
                    ->where('month', $monthLabel)
                    ->where('status', 'paid')
                    ->exists();
                if ($alreadyPaid) {
                    return;
                }

                /** @var Saving|null $account */
                $account  = $customer->savings->first();
                $branchId = $customer->branch_id ?? null;

                // Refs (use your trait helpers)
                $primaryRef = $this->generatetrnxref('PRF');  // record ref
                $glRef      = $this->generatetrnxref('PRF');  // GL record ref

                // Balance basis
                $balance = 0.0;
                if ($account) {
                    $balanceStr = (string) ($account->account_balance ?? '0');
                    $balance = (float) preg_replace('/[^\d.\-]/', '', $balanceStr);
                }

                // Profit
                $profit = round($balance * $monthlyRate, 2);

                // Create record first (audit/reporting)
                $record = MonthlySavingsProfit::create([
                    'customer_id'           => $customer->id,
                    'user_id'               => $userId,
                    'branch_id'             => $branchId,
                    'account_number'        => $customer->acctno,
                    'interest_rate_percent' => $interestRate,
                    'balance_basis'         => $balance,
                    'profit_amount'         => $profit,
                    'withdrawals_total'     => (int) ($customer->withdrawals_total ?? 0),
                    'status'                => 'pending',
                    'reference'             => $primaryRef,
                    'month'                 => $monthLabel,
                    'initiated_by'          => $actor,
                    'note'                  => null,
                ]);

                // Skip crediting if no account or zero profit (leave as pending with note) — DO NOT add to payload
                if (!$account || $profit <= 0) {
                    $record->note = !$account
                        ? 'No savings account found'
                        : 'Zero profit — not paid';
                    $record->save();
                    return;
                }

                // ===== PAY THE PROFIT =====

                // Update savings balances
                $newBal = $balance + $profit;
                $account->account_balance = $newBal;
                $account->ledger_balance  = $newBal;
                $account->save();

                // Create savings transaction (credit interest)
                $this->create_saving_transaction(
                    $userId,                        // user_id
                    $customer->id,                  // customer_id
                    $branchId,                      // branch_id
                    $profit,                        // amount
                    'interest',                     // type
                    'SYSTEM',                       // device
                    '0',                            // system_interest
                    null,                           // slip
                    null,                           // is_approve
                    'internal',                     // transfer_type / channel
                    $customer->acctno,              // destination_account
                    $primaryRef,                    // reference_no
                    "Monthly Savings Profit ({$interestRate}%)",
                    'approved',                     // status
                    '1',                            // status_type
                    'trnsfer',                      // trnx_type
                    $actor                          // initiated_by / approve_by
                );

                // GL: debit Interest Expense GL
                $expenseGL = GeneralLedger::where('gl_code', self::INTEREST_EXPENSE_GL)
                    ->lockForUpdate()
                    ->firstOrFail();

                $this->gltransaction('withdrawal', $expenseGL, $profit, null);

                $this->create_saving_transaction_gl(
                    $userId,                        // user_id
                    $expenseGL->id,                 // general_ledger_id
                    $branchId,                      // branch_id
                    $profit,                        // amount
                    'debit',                        // type
                    'SYSTEM',                       // device
                    $primaryRef,                    // slip
                    $glRef,                         // reference_no
                    "Monthly Savings Profit EXPENSE (DR) for {$customer->acctno}",
                    'approved',                     // status
                    $actor                          // initiated_by
                );

                // Mark paid
                $record->status = 'paid';
                $record->note   = 'Profit paid and posted';
                $record->save();

                // Audit
                $this->tracktrails(
                    $userId,
                    $branchId,
                    $actor,
                    'Monthly Savings Profit',
                    "Paid {$profit} to {$customer->acctno} for {$monthLabel} (ref {$primaryRef})"
                );

                // Only push PAID rows to payload
               return $payload[] = [
                    'customer_id'       => $customer->id,
                    'name'              => trim(($customer->first_name ?? '') . ' ' . ($customer->last_name ?? '')) ?: ($customer->name ?? null),
                    'account_number'    => $customer->acctno,
                    'withdrawals_total' => (int) ($customer->withdrawals_total ?? 0),
                    'balance'           => round($balance, 2),
                    'rate_used_percent' => $interestRate . '%',
                    'monthly_profit'    => $profit,
                    'reference'         => $primaryRef,
                    'month'             => $monthLabel,
                    'status'            => 'paid',
                ];
            });
        }

        return array_values(array_filter($payload, fn ($r) => ($r['status'] ?? '') === 'paid'));
    }
}
