<?php

namespace App\Services;

use App\Models\Customer;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Throwable;

class MatrixPayService
{
    protected string $baseUrl;
    protected string $apiKey;
    protected string $secretKey;
    protected string $settlementAccount;

    public function __construct()
    {
        $this->baseUrl = config('matrixpay.base_url');
        $this->apiKey = config('matrixpay.api_key');
        $this->secretKey = config('matrixpay.secret_key');
        $this->settlementAccount = config('matrixpay.settlement_account');
    }

    public function createAccount(Customer $customer)
    {
        $payload = [
            'accountName' => "{$customer->first_name} {$customer->last_name}",
            'accountNumber' => $customer->acctno,
            'bvn' => $customer->bvn,
            'BusinessName' => $customer->business_name,
            'enable_business_name' => !empty($customer->business_name) ? 1 : 0,
            'SettlementAccountNumber' => $this->settlementAccount,
        ];

        try {
            Log::info('Matrix payload', $payload);

            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'publickey' => $this->apiKey,
                'secretkey' => $this->secretKey,
            ])
                ->timeout(60)
                ->post($this->baseUrl . '/virtual-account/create', $payload);

            $response = $response->json();
            Log::info('Matrix Pay Create Account', $response);

            Customer::whereKey($customer->id)->update([
                'external_account_provider' => 'matrixpay',
                'external_account_status' => $response['status'] === true ? 1 : 0,
                'external_account_attempts' => DB::raw('external_account_attempts + 1'),
                'external_account_payload' => json_encode($payload),
                'external_account_response' => json_encode($response),
            ]);
        } catch (Throwable $e) {
            Log::error('Matrix error', ['error' => $e->getMessage()]);
        }
    }


    public function verifyBVN($bvn)
    {
        $payload = [
            'bvn' => $bvn,
        ];

        Log::info('Matrix BVN Payload', $payload);

        try {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'publickey' => $this->apiKey,
                'secretkey' => $this->secretKey,
            ])
                ->timeout(120)
                ->post($this->baseUrl . '/validation/bvn-verify', $payload);

            Log::info('Matrix BVN Response', $response->json());
            return $response->json();
        } catch (Throwable $e) {
            Log::error('Matrix BVN error', ['error' => $e->getMessage()]);
            return null;
        }
    }

    public function updateAccount(Customer $customer)
    {
        $payload = [
            'accountName' => "{$customer->first_name} {$customer->last_name}",
            'accountNumber' => $customer->acctno,
            'bvn' => $customer->bvn,
            'BusinessName' => $customer->business_name,
            'enable_business_name' => !empty($customer->business_name) ? 1 : 0,
            'SettlementAccountNumber' => $this->settlementAccount,
        ];

        try {
            Log::info('Matrix Update Account Payload', $payload);

            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'publickey' => $this->apiKey,
                'secretkey' => $this->secretKey,
            ])
                ->timeout(120)
                ->post($this->baseUrl . '/virtual-account/update', $payload);

            $response = $response->json();
            Log::info('Matrix Pay Update Account', $response);
        } catch (Throwable $e) {
            Log::error('Matrix error', ['error' => $e->getMessage()]);
        }
    }


    public function getBbanks($data)
    {

        try {
            Log::info('Matrix Get Bank Listd Request sent');

            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'publickey' => $this->apiKey,
                'secretkey' => $this->secretKey,
            ])
                ->timeout(120)
                ->get($this->baseUrl . '/get-banks');

            $response = $response->json();
            Log::info('Matrix Pay Get Bank List', $response);
        } catch (Throwable $e) {
            Log::error('Matrix error', ['error' => $e->getMessage()]);
        }
    }


    public function transfer($data)
    {
        $payload = [
            'beneficiaryBankCode' => $data['bank_code'],
            'beneficiaryAccountNumber' =>  $data['account_number'],
            'beneficiaryName' =>  $data['account_name'],
            'amount' =>  $data['amount'],
            'narration' => $data['narration'],
            'sourceAccountNumber' =>  $data['source_account'],
            'paymentReference' =>  $data['payment_reference'],
            'nameEnquiryRef' => $data['name_enquiry_ref'],
        ];

        try {
            Log::info('MatrixPay Transfer Payload', $payload);
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'publickey' => $this->apiKey,
                'secretkey' => $this->secretKey,
            ])
                ->timeout(120)
                ->post($this->baseUrl . '/transactions/bank/account-transfer', $payload);

            $response = $response->json();
            Log::info('MatrixPay Transfer Response', $response);
        } catch (Throwable $e) {
            Log::error('Matrix error', ['error' => $e->getMessage()]);
        }
    }


    public function TSQ($data)
    {
        $payload = [
            'settlementAccountNumber' => $data['bank_code'],
            'paymentReference' =>  $data['account_number']
        ];

        try {
            Log::info('MatrixPay TSQ Payload', $payload);
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'publickey' => $this->apiKey,
                'secretkey' => $this->secretKey,
            ])
                ->timeout(120)
                ->post($this->baseUrl . '/transactions/bank/account-transfer', $payload);

            $response = $response->json();
            Log::info('MatrixPay TSQ Response', $response);
        } catch (Throwable $e) {
            Log::error('MatrixPay TSQL error', ['error' => $e->getMessage()]);
        }
    }
}
