<?php

namespace App\Http\Controllers;

use Throwable;
use App\Models\Saving;
use App\Models\Customer;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Excel;
use App\Models\GeneralLedger;
use App\Models\TargetSavings;
use App\Models\Accountofficer;
use App\Http\Traites\AuditTraite;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Exports\TargetSavingsExport;
use App\Models\TargetSavingsProduct;
use Illuminate\Support\Facades\Auth;
use App\Models\TargetSavingsSchedule;
use App\Services\TargetSavingsService;
use Spatie\SimpleExcel\SimpleExcelWriter;




class TargetSavingsController extends Controller
{
    use AuditTraite;

    public function __construct(private TargetSavingsService $targetSavingsService)
    {
    }

    public function manage_product()
    {
        $products = TargetSavingsProduct::latest()->paginate(50);
        return view('target_savings.product.manage_product', ['products' => $products]);
    }


    public function create_product()
    {
        return view('target_savings.product.create_product');
    }

    public function store_product(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'minimum_target_deposit' => 'required|numeric|min:5000',
            'interest_rate' => 'required|numeric|min:1',
            'minimum_hold_days' => 'required|integer|min:30',
            'description' => 'nullable|string',
        ]);

        $this->targetSavingsService->createProduct($validated);

        return redirect()->route('targetsavings.product')->with('success', 'Target Savings Product created successfully.');
    }

    private function generateUniqueCode(int $length = 10)
    {
        do {
            $code = '';
            for ($i = 0; $i < $length; $i++) {
                $code .= (string) random_int(0, 10);
            }
        } while (TargetSavingsProduct::where('product_code', $code)->exists());

        return $code;
    }

    public function edit_product($id)
    {
        $product = TargetSavingsProduct::findOrFail($id);
        return view('target_savings.product.edit_product', ['product' => $product]);
    }

    public function update_product(Request $request, int $id)
    {
        $validated = $request->validate([
            'name' => 'nullable|string|max:255',
            'minimum_target_deposit' => 'nullable|numeric|min:5000',
            'interest_rate' => 'nullable|numeric|min:1',
            'minimum_hold_days' => 'nullable|integer|min:30',
            'description' => 'nullable|string',
        ]);

        $this->targetSavingsService->updateProduct($id, $validated);

        return redirect()->route('targetsavings.product')->with('success', 'Update successfully.');
    }


    public function manage_targetsavings(Request $request)
    {
        $q = trim((string) $request->input('q', ''));

        $query = TargetSavings::query()->with('customer');

        if ($q !== '') {
            $query->where(function ($qb) use ($q) {
                $qb->where('code', 'like', "%{$q}%")
                    ->orWhere('account_number', 'like', "%{$q}%")
                    ->orWhere('frequency', 'like', "%{$q}%")
                    ->orWhere('status', 'like', "%{$q}%")
                    ->orWhere('target_amount', 'like', "%{$q}%")
                    ->orWhere('auto_save_amount', 'like', "%{$q}%")
                    ->orWhereHas('customer', function ($qc) use ($q) {
                        $qc->where('first_name', 'like', "%{$q}%")
                            ->orWhere('last_name', 'like', "%{$q}%")
                            ->orWhereRaw("CONCAT(first_name,' ',last_name) LIKE ?", ["%{$q}%"]);
                    });
            });
        }

        $targets = $query->latest()->paginate(100)->withQueryString();

        return view('target_savings.manage_targetsavings', [
            'targets' => $targets,
            'q' => $q,
        ]);
    }


    public function export_targetsavings(Request $request)
    {
        try {
            $type = strtolower((string) $request->query('type', 'excel'));
            $q = trim((string) $request->query('q', ''));
            $status = strtolower((string) $request->query('status', '')); // optional

            $rows = $this->buildExportQuery($q, $status)->get();

            switch ($type) {
                case 'excel':
                case 'xlsx':
                    return $this->exportExcel($rows);

                case 'copy':
                    return response()->view('target_savings.exports.copy', [
                        'rows' => $rows,
                        'q' => $q,
                    ]);

                case 'print':
                    return response()->view('target_savings.exports.print', [
                        'rows' => $rows,
                        'q' => $q,
                    ]);

                default:
                    return back()->with('error', 'Unknown export type. Use excel, copy, or print.');
            }
        } catch (Throwable $e) {
            Log::error('targetsavings export failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return back()->with('error', 'Unable to export at the moment.');
        }
    }

    private function buildExportQuery(string $q, string $status)
    {
        $allowedStatuses = ['pending', 'approved', 'active', 'declined', 'rejected', 'terminate', 'closed', 'close'];

        return TargetSavings::query()
            ->leftJoin('customers', 'customers.id', '=', 'target_savings.customer_id')
            ->when($status !== '' && in_array($status, $allowedStatuses, true), function ($qb) use ($status) {
                $qb->where('target_savings.status', $status);
            })
            ->when($q !== '', function ($qb) use ($q) {
                $qb->where(function ($w) use ($q) {
                    $w->where('target_savings.code', 'like', "%{$q}%")
                        ->orWhere('target_savings.account_number', 'like', "%{$q}%")
                        ->orWhere('target_savings.frequency', 'like', "%{$q}%")
                        ->orWhere('target_savings.status', 'like', "%{$q}%")
                        ->orWhere('target_savings.target_amount', 'like', "%{$q}%")
                        ->orWhere('target_savings.auto_save_amount', 'like', "%{$q}%")
                        ->orWhere(DB::raw("CONCAT(COALESCE(customers.first_name,''),' ',COALESCE(customers.last_name,''))"), 'like', "%{$q}%");
                });
            })
            ->select([
                'target_savings.id',
                'target_savings.code',
                'target_savings.account_number',
                'target_savings.target_amount',
                'target_savings.auto_save_amount',
                'target_savings.savings_plan_name',
                'target_savings.frequency',
                'target_savings.auto_save_start_date',
                'target_savings.status',
                DB::raw("COALESCE(customers.first_name, '') as c_first_name"),
                DB::raw("COALESCE(customers.last_name,  '') as c_last_name"),
            ])
            ->orderByDesc('target_savings.id');
    }


    private function exportExcel($rows)
    {
        $filename = 'target_savings_' . now()->format('Ymd_His') . '.xlsx';
        $path = storage_path('app/tmp/' . $filename);

        if (!is_dir(dirname($path))) {
            mkdir(dirname($path), 0775, true);
        }

        $writer = SimpleExcelWriter::create($path)
            ->addHeader(['Code', 'Name', 'Account No', 'Target Amount', 'Save Amount', 'Target Name', 'Frequency', 'Start Date', 'Status']);

        foreach ($rows as $r) {
            $name = trim(ucwords(trim(($r->c_last_name ?? '') . ' ' . ($r->c_first_name ?? '')))) ?: 'N/A';
            $writer->addRow([
                'Code' => $r->code,
                'Name' => $name,
                'Account No' => $r->account_number,
                'Target Amount' => number_format((float) $r->target_amount, 2, '.', ''),
                'Save Amount' => number_format((float) $r->auto_save_amount, 2, '.', ''),
                'Target Name' => $r->savings_plan_name,
                'Frequency' => ucfirst((string) $r->frequency),
                'Start Date' => $r->auto_save_start_date,
                'Status' => ucfirst((string) $r->status),
            ]);
        }
        $writer->close();

        $contents = file_get_contents($path);
        @unlink($path);

        return response($contents, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            'Cache-Control' => 'no-store, no-cache, must-revalidate',
            'Pragma' => 'no-cache',
        ]);
    }



    public function pending_targetsavings(Request $request)
    {
        $q = trim((string) $request->input('q', ''));

        $query = TargetSavings::query()
            ->with('customer')
            ->where('status', 'pending');

        if ($q !== '') {
            $query->where(function ($qb) use ($q) {
                $qb->where('code', 'like', "%{$q}%")
                    ->orWhere('account_number', 'like', "%{$q}%")
                    ->orWhere('frequency', 'like', "%{$q}%")
                    ->orWhere('status', 'like', "%{$q}%")
                    // allow searching amounts as plain digits/decimals
                    ->orWhere('target_amount', 'like', "%{$q}%")
                    ->orWhere('auto_save_amount', 'like', "%{$q}%")
                    ->orWhereHas('customer', function ($qc) use ($q) {
                        $qc->where('first_name', 'like', "%{$q}%")
                            ->orWhere('last_name', 'like', "%{$q}%")
                            ->orWhereRaw("CONCAT(first_name,' ',last_name) LIKE ?", ["%{$q}%"]);
                    });
            });
        }

        $targets = $query->latest()->paginate(100)->withQueryString();

        return view('target_savings.pending_targetsavings', [
            'targets' => $targets,
            'q' => $q,
        ]);
    }






    public function create_targetsavings()
    {
        $products = TargetSavingsProduct::all();
        return view('target_savings.create_targetsavings', ['products' => $products]);
    }



    public function store_targetsavings(Request $request)
    {
        $isAuto = $request->boolean('auto_save');

        if ($isAuto) {
            $validated = $request->validate([
                'target_savings_products_id' => ['required', 'exists:target_savings_products,id'],
                'acctno' => ['required', 'digits:10', 'exists:customers,acctno'],
                'customer_id' => ['required', 'exists:customers,id'],
                'customer_name' => ['required', 'string', 'max:255'],
                'account_number' => ['required', 'string', 'exists:customers,acctno'],

                'savings_plan_name' => ['required', 'string', 'max:255'],
                'target_amount' => ['required', 'numeric', 'min:0.01'],

                'auto_save' => ['required', 'boolean'],
                'auto_save_start_date' => ['required', 'date', 'after_or_equal:today'],
                'auto_save_amount' => ['required', 'numeric', 'min:0.01'],
                'frequency' => ['required', 'in:daily,weekly,monthly'],
            ]);

            if ($validated['auto_save_amount'] > $validated['target_amount']) {
                return back()
                    ->withInput()
                    ->with('error', 'Auto-save amount cannot be higher than the target amount.');
            }

        } else {
            $validated = $request->validate([
                'target_savings_products_id' => ['required', 'exists:target_savings_products,id'],
                'acctno' => ['required', 'digits:10', 'exists:customers,acctno'],
                'customer_id' => ['required', 'exists:customers,id'],
                'customer_name' => ['required', 'string', 'max:255'],
                'account_number' => ['required', 'string', 'exists:customers,acctno'],

                'savings_plan_name' => ['required', 'string', 'max:255'],
                'target_amount' => ['required', 'numeric', 'min:0.01'],
            ]);

            $validated['auto_save'] = 0;
        }

        try {
            if ($isAuto) {
                // Auto-Save path
                $this->targetSavingsService->storeAutoSaveTargetSavings($validated);
            } else {
                // Non-Auto path
                $this->targetSavingsService->storeTargetsavings($validated);
            }

            return redirect()->route('targetsavings.manage')
                ->with('success', 'Target Savings created successfully');
        } catch (Throwable $e) {
            Log::error('Target savings create failed', ['err' => $e->getMessage()]);
            return back()->withInput()->with('error', $e->getMessage() ?: 'Unable to create Target Savings');
        }
    }




    public function show_targetsavings($id)
    {

        $target = TargetSavings::findOrFail($id);

        $customer = Customer::where('id', $target->customer_id)->first();
        $saving = Saving::where('customer_id', $customer->id)->first();
        $accountOfficer = Accountofficer::where('id', $customer->accountofficer_id)->first();
        $schedules = TargetSavingsSchedule::where('target_savings_id', $target->id)
            ->where('target_savings_code', $target->code)
            ->get(); // <-- use get(), not all()

        //    return $schedule = TargetSavingsSchedule::all();


        $customer['account_balance'] = $saving['account_balance'] ?? 0;
        $customer['ledger_balance'] = $saving['ledger_balance'] ?? 0;
        $customer['account_officer'] = $accountOfficer['full_name'] ?? null;

        // product key is singular in DB
        $product = TargetSavingsProduct::findOrFail($target->target_savings_product_id);

        // return [
        //     'target' => $target,
        //                 'customer' => $customer,
        //     'accountOfficer' => $accountOfficer,
        //     'product' => $product,

        // ];
        return view('target_savings.show_targetsavings', [
            'target' => $target,
            'customer' => $customer,
            'accountOfficer' => $accountOfficer,
            'product' => $product,
            'schedules' => $schedules
        ]);
    }

    public function edit_targetsavings($id)
    {
        $target = TargetSavings::findOrFail($id);

        $products = TargetSavingsProduct::all(['id', 'name', 'minimum_target_deposit', 'interest_rate', 'minimum_hold_days', 'description']);
        $customer = Customer::findOrFail($target->customer_id, ['id', 'first_name', 'last_name', 'acctno']);

        $balance = Saving::where('customer_id', $customer->id)->value('account_balance');
        $customer->account_balance = $balance ?? 0;

        return view('target_savings.edit_targetsavings', [
            'target' => $target,
            'products' => $products,
            'customer' => $customer,
        ]);
    }

    public function update_targetsavings(Request $request, $id)
    {
        $validated = $request->validate([
            // still posted as plural from the form; we’ll map inside the service
            'target_savings_products_id' => ['required', 'exists:target_savings_products,id'],
            'customer_id' => ['required', 'exists:customers,id'],
            'account_number' => ['required', 'exists:customers,acctno'],
            'savings_plan_name' => ['required', 'string', 'max:255'],
            'target_amount' => ['required', 'numeric', 'min:0.01'],
            'auto_save' => ['nullable', 'boolean'],
            'auto_save_start_date' => ['required', 'date'],
            'auto_save_amount' => ['required', 'numeric', 'min:0'],
            'frequency' => ['required', 'in:daily,weekly,monthly'],
        ]);

        try {
            $this->targetSavingsService->updateTargetsavings((int) $id, $validated);
            return redirect()->route('targetsavings.manage')->with('success', 'Target Savings updated successfully.');
        } catch (Throwable $e) {
            Log::error('Target savings update failed', ['err' => $e->getMessage()]);
            return back()->withInput()->with('error', $e->getMessage() ?: 'Unable to update Target Savings');
        }
    }

    public function get_customer_details($account)
    {
        return $this->targetSavingsService->getCustomer($account);
    }

    public function get_product_details($id)
    {
        return $this->targetSavingsService->getProducteDetails($id);
    }

    public function approve_targetsavings($id)
    {

        return $this->targetSavingsService->approveAndDebitTargetSavings($id);
    }

    public function reject_targetsavings($id)
    {
        return $this->targetSavingsService->rejectTargetsavings($id);
    }

    public function terminate_targetsavings($id)
    {
        return $this->targetSavingsService->terminateTargetsavings($id);
    }

    public function disbursed_targetsavings($id)
    {
        return $this->targetSavingsService->disbursedTargetSavings($id);
    }
}
