<?php

namespace App\Http\Controllers\Api;

use App\Models\Bank;
use App\Mail\AppMail;
use App\Models\Email;
use App\Models\Charge;
use App\Models\Saving;
use App\Models\Setting;
use App\Models\Customer;
use App\Models\ProvidusKey;
use Faker\Factory as Faker;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\GeneralLedger;
use App\Traits\HttpResponses;
use Illuminate\Support\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Http\Traites\UserTraite;
use App\Http\Traites\AuditTraite;
use App\Http\Traites\SavingTraite;
use App\Models\SavingsTransaction;
use Illuminate\Support\Facades\DB;
use App\Models\NotificationPayload;
use App\Http\Controllers\Controller;
use App\Http\Traites\TransferTraite;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use Illuminate\Contracts\Cache\LockTimeoutException;

class TransactionController extends Controller
{
    use SavingTraite;
    use AuditTraite;
    use UserTraite;
    use TransferTraite;
    use HttpResponses;

    private $murl, $mapikey, $msercetkey, $macctno, $matrixurl, $matrixpkey, $matrixskey, $matrixacctno;

    public function __construct()
    {

        if (env('APP_MODE') == "test") {

            $this->murl = env('MONNIFY_SANDBOX_URL');
            $this->mapikey = env('MONNIFY_SANDBOX_API_KEY');
            $this->msercetkey = env('MONNIFY_SANDBOX_SECRET_KEY');
            $this->macctno = env('MONNIFY_SANDBOX_ACCOUNT_NUMBER');

            $this->matrixurl =  env('ASSETMATRIX_TEST_BASE_URL');
            $this->matrixpkey = env('ASSETMATRIX_TEST_PUBLIC_KEY');
            $this->matrixskey = env('ASSETMATRIX_TEST_ENCRYPT_KEY');
            $this->matrixacctno = env('ASSETMATRIX_TEST_SETTLEMENT_ACCOUNT');
        } else {

            $this->murl = env('MONNIFY_LIVE_URL');
            $this->mapikey = env('MONNIFY_LIVE_API_KEY');
            $this->msercetkey = env('MONNIFY_LIVE_SECRET_KEY');
            $this->macctno = env('MONNIFY_LIVE_ACCOUNT_NUMBER');

            $this->matrixurl =  env('ASSETMATRIX_BASE_URL');
            $this->matrixpkey = env('ASSETMATRIX_LIVE_PUBLIC_KEY');
            $this->matrixskey = env('ASSETMATRIX_LIVE_ENCRYPT_KEY');
            $this->matrixacctno = env('ASSETMATRIX_LIVE_SETTLEMENT_ACCOUNT');
        }
    }

    public function getAllBanks()
    {
        $banknames = [];
        $banks = Bank::orderBy('bank_name', 'ASC')->get();
        foreach ($banks as $bank) {
            array_push($banknames, ["bank_name" => $bank["bank_name"], "bank_code" => $bank["bank_code"], "bank_logo" => url('/') . '/' . $bank["bank_logo"]]);
        }
        return $this->success('Banks Fetched Successfully', $banknames);
    }

    public function get_transactionWebsock()
    {
        $transactions = SavingsTransaction::where('customer_id', Auth::id())
            ->orderBy('created_at', 'DESC')
            ->get();

        $this->logInfo("transaction history websocket init", $transactions);

        $data = [
            'transactions' => $transactions,
            'channel'      => 'private-transactions.' . Auth::id(),
            'event'        => 'TransactionCreated',
        ];

        if ($transactions->isEmpty()) {
            return $this->success('No transaction found', $data);
        }

        // event(new TransactionCreated($transaction));

        return $this->success('Transactions retrieved successfully', $data);
    }

    public function verifyBankAccount(Request $request)
    {
        $this->logInfo("validating Bank Account", $request->all());
        $validation = Validator::make($request->all(), [
            "account_number" => "required|numeric|digits:10",
            "bank_code" => "required|numeric",
        ]);

        if ($validation->fails()) {
            return $this->error('Validation error.', 422, $validation->errors()->toArray());
        }
        $faker = Faker::create();

        $data = [
            'first_name' => $faker->firstName,
            'last_name'  => $faker->lastName,
            'bankCode'   => $request->bank_code,
        ];

        return $this->success('Bank Account Verified Successfully', $data);
        // $baseurl = env('APP_MODE') == "test" ? env("TEST_ACCESS_URL") : env("LIVE_ACCESS_URL");

        // $token = $this->AccessMiddelware();

        // $response =  Http::withHeaders([
        //     "Content-Type" => "application/json",
        //     "Authorization" => "Bearer " . $token['token'],
        // ])->post($baseurl . "outward/name/enquiry", [
        //     "destination_institution_code" => $r->bank_code,
        //     "account_number" => $r->account_number,
        //     "channel_code" => "2"
        // ])->json();

        // if ($response["data"]["response"]["ResponseCode"] == '00') {

        //     $vdata = [
        //         "first_name" => $response["data"]["response"]["AccountName"],
        //         "last_name" => "",
        //         "bankCode" => $r->bank_code
        //     ];

        //     $this->logInfo("Bank Account verified", $vdata);

        //     return response()->json(['status' => true, 'message' => 'Bank Account Verified Successfully', 'data' => $vdata]);
        // } else {
        //     return response()->json(['status' => false, 'message' => 'Failed to Verify Bank Account']);
        // }
    }

    public function get_transactionHistory()
    {
        $transactions = SavingsTransaction::where('customer_id', Auth::id())
            ->orderBy('created_at', 'DESC')
            ->get();

        $this->logInfo("transaction history", $transactions);

        if ($transactions->isEmpty()) {
            return $this->success('No transaction found');
        }

        return $this->success('Transactions retrieved successfully', $transactions);
    }

    public function single_transaction($ref)
    {
        $transaction = SavingsTransaction::where('customer_id', Auth::id())->where('reference_no', $ref)
            ->orderBy('created_at', 'DESC')
            ->get();

        $this->logInfo("transaction history", $transaction);

        if ($transaction->isEmpty()) {
            return $this->success('No transaction found');
        }

        return $this->success('Transaction retrieved successfully', $transaction);
    }


    public function get_transactionStatement()
    {
        $transac = SavingsTransaction::where('customer_id', Auth::user()->id)
            ->whereBetween('created_at', [request()->fromdate, request()->todate])
            ->get();

        return response()->json(['status' => true, 'message' => 'Transaction Statement Fetched', 'data' => $transac]);
    }


    public function send_transactionStatement(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'start_date' => 'required|date_format:d-m-Y',
                'end_date'   => 'required|date_format:d-m-Y|after_or_equal:start_date',
            ]);

            if ($validator->fails()) {
                return $this->error(
                    'Validation error',
                    422
                );
            }

            $user = Auth::user();
            if (! $user) {
                return $this->error('Unauthenticated', 401);
            }

            $from = Carbon::createFromFormat('d-m-Y', $request->start_date)->startOfDay();
            $to   = Carbon::createFromFormat('d-m-Y', $request->end_date)->endOfDay();

            $transactions = SavingsTransaction::where('customer_id', $user->id)
                ->whereBetween('created_at', [$from, $to])
                ->orderBy('created_at', 'ASC')
                ->get();

            if ($transactions->isEmpty()) {
                return $this->error(
                    'No transactions found for the selected date range',
                    404
                );
            }

            $pdfBinary = Pdf::loadView('pdf.account-statement', [
                'user'         => $user,
                'transactions' => $transactions,
                'from'         => $from,
                'to'           => $to,
            ])->output();

            $fileName = 'Account_Statement_' .
                $from->format('Ymd') . '_' .
                $to->format('Ymd') . '.pdf';

            $title = 'Account Statement';
            $body  = 'Please find attached your account statement from ' .
                $from->format('d-m-Y') . ' to ' . $to->format('d-m-Y') . '.';

            Mail::to($user->email)->queue(
                new AppMail($title, $body, $pdfBinary, $fileName)
            );

            Email::create([
                'uuid'      => Str::uuid(),
                'user_id'   => $user->id,
                'subject'   => $title,
                'message'   => $body,
                'recipient' => $user->email,
                'branch_id' => 1,
            ]);

            return $this->success(
                'Account statement generated and sent to your email successfully'
            );
        } catch (\Exception $e) {
            if (method_exists($this, 'logError')) {
                $this->logError('send_transactionStatement error', [
                    'message' => $e->getMessage(),
                    'trace'   => $e->getTraceAsString(),
                ]);
            }

            return $this->error(
                'Unable to generate account statement at the moment. Please try again later.',
                500
            );
        }
    }



    public function initiateTransaction(Request $request)
    {
        $user   = Auth::user();
        $userId = $user->id;
        $acctNo = $user->acctno;
        $userName = trim($user->last_name . ' ' . $user->first_name);

        $lock = Cache::lock('initrnx-' . $userId, 2);

        try {
            $lock->block(1);

            $this->logInfo('initializing transaction', $request->all());

            /** Validate request */
            $validation = Validator::make($request->all(), [
                'amount'           => 'required|numeric|gt:0',
                'transaction_type' => 'required|string',
                'description'      => 'required|string',
                'platform'         => 'required|string'
            ]);

            if ($validation->fails()) {
                return $this->error(
                    $validation->errors()->first(),
                    '406'
                );
            }

            /** Prevent special characters in narration */
            if (preg_match('/[\'^£$%&*()}{@#~?><>,|=_+¬-]/', $request->description)) {
                return $this->error(
                    'No special character allowed in narration',
                    '406'
                );
            }

            $this->tracktrails(
                '1',
                '1',
                $userName,
                'customer',
                'transaction initialized'
            );

            $settings = new Setting();

            $amount      = (float) $request->amount;
            $charge      = $this->GetTransactionCharge((int) $amount);
            $totalAmount = $amount + $charge;

            $this->logInfo('transaction charge', ['charge' => $charge]);

            /** Validate company balance */
            $companyBalance = $this->validatecompanybalance($totalAmount, 'combal');
            if ($companyBalance['status'] === false) {
                return $this->error(
                    $companyBalance['message'] ?? 'Company balance validation failed',
                    '406'
                );
            }

            /** Check customer restriction */
            if ($this->checkCustomerRestriction($userId)) {
                $this->tracktrails(
                    '1',
                    '1',
                    $userName,
                    'customer',
                    'Account Restricted'
                );

                return $this->error(
                    'Your account has been restricted. Please contact support.',
                    '403'
                );
            }

            /** Check lien */
            $lien = $this->checkCustomerLienStatus($userId);
            if ($lien['status'] === true && $lien['lien'] == 2) {
                return $this->error(
                    'Your account has been lien (' . $lien['messages'] . '). Please contact support.',
                    '403'
                );
            }

            /** Validate customer balance */
            $balanceCheck = $this->validatecustomerbalance($userId, $totalAmount);
            if ($balanceCheck['status'] === false) {
                return $this->error(
                    $balanceCheck['message'],
                    '406'
                );
            }

            /** Validate online transfer limits */
            $onlineLimit = $this->validateOnlineTransfer($amount, $userId);
            if ($onlineLimit['status'] === false) {
                return $this->error(
                    $onlineLimit['message'],
                    '406'
                );
            }

            $transferLimit = $this->validateTransfer(
                $amount,
                $settings->getsettingskey('online_transfer'),
                userid: $userId
            );

            if ($transferLimit['status'] === false) {
                return $this->error(
                    $transferLimit['message'],
                    '406'
                );
            }

            /** Final lien check */
            $customerLien = $this->CheckCustomerLienAccount($userId, $totalAmount);
            if ($customerLien['status'] === false) {
                return $this->error(
                    $customerLien['message'],
                    '403'
                );
            }

            /** Handle transaction types */
            if ($request->transaction_type === 'bank') {

                $trnxid = $this->generatetrnxref('bnk');

                return $this->success(
                    'Transaction initialized',
                    [
                        'charge'                => $charge,
                        'platform'              => $request->platform,
                        'transaction_reference' => $trnxid,
                        'transactionTypeId'     => $request->transaction_type,
                        'transfer_amount'       => $amount,
                        'total'                 => $totalAmount
                    ],
                    '000'
                );
            }

            if ($request->transaction_type === 'wallet') {

                if ($acctNo === $request->destination_account) {
                    return $this->error(
                        'Cannot transfer to self',
                        '406'
                    );
                }

                $trnxid = $this->generatetrnxref('w');

                return $this->success(
                    'Transaction initialized',
                    [
                        'charge'                => 0,
                        'platform'              => $request->platform,
                        'transaction_reference' => $trnxid,
                        'transactionTypeId'     => $request->transaction_type,
                        'transfer_amount'       => $amount,
                        'total'                 => $amount
                    ],
                    '000'
                );
            }

            return $this->error(
                'Invalid transaction type',
                '406'
            );
        } catch (LockTimeoutException $e) {

            return $this->error(
                'Transaction is already in progress. Please try again.',
                '409'
            );
        } catch (\Throwable $e) {

            $this->logError('Transaction init error', $e);

            return $this->error(
                'Error processing transaction',
                '500'
            );
        } finally {
            optional($lock)->release();
        }
    }



    public function transferToBankAccount(Request $request)
    {

        $lock = Cache::lock('bnktrnx-' . Auth::user()->id, 2);

        try {

            $lock->block(1);

            // return $request->all();
            $this->logInfo("bank transfer", $request->except('transaction_pin'));

            $validation = Validator::make($request->all(), [
                "amount" => "required|numeric|gt:0|max:10000000",
                "destination_account" => "required",
                "transaction_reference" => "required|string",
                "receipient_name" => "required|string",
                "bank_code" => "required",
                "transaction_pin" => "required|numeric|gt:0",
                "platform" => "required|string"
            ]);


            if ($validation->fails()) {
                $ra = array("status" => false, "message" => $validation->errors()->all()[0]);
                return response()->json($ra, 406);
            }

            DB::beginTransaction();


            $usern = Auth::user()->last_name . " " . Auth::user()->first_name;

            $this->tracktrails('1', '1', $usern, 'customer', 'Transfer to Bank Via payout');

            $getsetvalue = new Setting();


            $description = $request->description;

            $tcharge = $this->GetTransactionCharge((int)$request->amount);
            // $tcharge = Charge::select('amount')->where('id',$getsetvalue->getsettingskey('transfer_charge'))->first();
            // $ocharge = Charge::select('amount')->where('id',$getsetvalue->getsettingskey('othercharges'))->first();
            // $bankcharger = $getsetvalue->getsettingskey('bankcharge');
            // $charge = $tcharge->amount + $bankcharger + $ocharge->amount;
            // $totalAmount = $request->amount + $charge;
            // $tchargeamt = $tcharge->amount + $bankcharger;

            $charge = $tcharge;

            $totalAmount = $request->amount + $charge;
            $tchargeamt = $tcharge;

            $compbal = $this->validatecompanybalance($totalAmount, "combal");
            if ($compbal["status"] == false) {

                $this->logInfo("validating company balance", $compbal);

                return response()->json($compbal, 406);
            }

            //validate matrixpay balance
            //  $matrixpaybal = $this->validateMatixpayBalance($this->matrixacctno,$request->amount);
            //  if($matrixpaybal['status'] == false){
            //  return response()->json($matrixpaybal,406);
            //  }

            $chkcres = $this->checkCustomerRestriction(Auth::user()->id);
            if ($chkcres == true) {

                $this->tracktrails('1', '1', $usern, 'customer', 'Account Restricted');

                $this->logInfo("", "Customer Account Restricted");

                return response()->json(['status' => false, 'message' => 'Your Account Has Been Restricted. Please contact support'], 406);
            }

            $chklien = $this->checkCustomerLienStatus(Auth::user()->id);
            if ($chklien['status'] == true && $chklien['lien'] == 2) {


                $this->tracktrails('1', '1', $usern, 'customer', 'Account has been lien');

                $this->logInfo("Account lien", $chklien);

                return response()->json(['status' => false, 'message' => 'Your Account Has Been Lien(' . $chklien['messages'] . ')...please contact support']);
            }

            $validateuserpin = $this->validatetrnxpin($request->transaction_pin, Auth::user()->id);
            if ($validateuserpin["status"] == false) {

                $this->tracktrails('1', '1', $usern, 'customer', $validateuserpin["message"]);

                $this->logInfo("Customer pin validation", $validateuserpin);

                return response()->json($validateuserpin, 406);
            }


            $validateuserbalance = $this->validatecustomerbalance(Auth::user()->id, $totalAmount);
            if ($validateuserbalance["status"] == false) {

                $this->tracktrails('1', '1', $usern, 'customer', $validateuserbalance["message"]);

                $this->logInfo("customer balance", $validateuserbalance);

                return response()->json($validateuserbalance, 406);
            }

            $validateOnlineTransferAmount = $this->validateOnlineTransfer($request->amount, Auth::user()->id);

            if ($validateOnlineTransferAmount['status'] == false) {

                $this->logInfo("online transfer limit", $validateOnlineTransferAmount);

                return response()->json($validateOnlineTransferAmount, 406);
            }

            $validateTransferAmount = $this->validateTransfer($request->amount, $getsetvalue->getsettingskey('online_transfer'), Auth::user()->id);

            if ($validateTransferAmount['status'] == false) {

                $this->logInfo("online transfer", $validateTransferAmount);

                return response()->json($validateTransferAmount, 406);
            }

            $customerlien = $this->CheckCustomerLienAccount(Auth::user()->id, $totalAmount);
            if ($customerlien['status'] == false) {
                return response()->json($customerlien, 406);
            }

            //initiate
            $this->create_saving_transaction(
                null,
                Auth::user()->id,
                null,
                $request->amount,
                'debit',
                $request->platform,
                '0',
                null,
                null,
                null,
                null,
                $request->transaction_reference,
                $description,
                'pending',
                '2',
                'trnsfer',
                ''
            );


            $transaction = SavingsTransaction::where('reference_no', $request->transaction_reference)->where('amount', $request->amount)->first();

            if ($transaction) {
                if ($transaction->status == "approved" || $transaction->status == "failed") {
                    return response()->json(["status" => false, 'message' => "Transaction has already been completed...Please Initiate Transaction"], 409);
                }
            }

            $glsavingdacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '20993097')->first();
            $glcurrentacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '20639526')->first();

            //transfer charges Gl
            $glaccttrr = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', $getsetvalue->getsettingskey('glcharges'))->first();

            if ($glaccttrr->status == '1') {
                $this->gltransaction('withdrawal', $glaccttrr, $tchargeamt, null);
                $this->create_saving_transaction_gl(null, $glaccttrr->id, null, $tchargeamt, 'credit', $request->platform, null, $this->generatetrnxref('trnxchrg'), 'transfer charges', 'approved', $usern);
            }

            //other charges Gl
            // $otherglacct = GeneralLedger::select('id','status','account_balance')->where('gl_code',$getsetvalue->getsettingskey('othrchargesgl'))->first();

            // if($otherglacct->status == '1'){
            // $this->gltransaction('withdrawal',$otherglacct,$ocharge->amount,null);
            // $this->create_saving_transaction_gl(null,$otherglacct->id,null, $ocharge->amount,'credit',$request->platform,null,$this->generatetrnxref('otc'),'others charges fees','approved',$usern);
            // }


            $debitCustomer = $this->DebitCustomerandcompanyGlAcct(Auth::user()->id, $totalAmount, $request->amount, $getsetvalue->getsettingskey('outwardoptiongl'), 'bnk', 'Bank Transfer via payout', $request->platform, $request->transaction_reference, $usern);

            if (Auth::user()->account_type == '1') { //saving acct GL

                if ($glsavingdacct->status == '1') {
                    $this->gltransaction('deposit', $glsavingdacct, $totalAmount, null);
                    $this->create_saving_transaction_gl(null, $glsavingdacct->id, null, $totalAmount, 'debit', $request->platform, $request->transaction_reference, $this->generatetrnxref('D'), 'customer debited', 'approved', $usern);
                }
            } elseif (Auth::user()->account_type == '2') { //current acct GL

                if ($glcurrentacct->status == '1') {
                    $this->gltransaction('deposit', $glcurrentacct, $totalAmount, null);
                    $this->create_saving_transaction_gl(null, $glcurrentacct->id, null, $totalAmount, 'debit', $request->platform, $request->transaction_reference, $this->generatetrnxref('D'), 'customer debited', 'approved', $usern);
                }
            }

            if (!$debitCustomer["status"]) {

                $this->updateTransactionAndAddTrnxcharges(
                    null,
                    Auth::user()->id,
                    null,
                    $charge,
                    'debit',
                    $request->platform,
                    '0',
                    null,
                    null,
                    null,
                    $request->transaction_reference,
                    "failed Transaction",
                    "failed Transaction",
                    'failed',
                    '10',
                    $usern,
                    ''
                );

                $this->tracktrails('1', '1', $usern, 'customer', 'Bank Transfer Failed');

                $reverd = $this->ReverseDebitTrnxandcompanyGlAcct(Auth::user()->id, $totalAmount, $request->amount, $request->transaction_reference, $getsetvalue->getsettingskey('outwardoptiongl'), 'bnk', 'Transaction reversed', $request->platform, 'trnsfer', $usern, '');

                //reverse transfer charges Gl
                if ($glaccttrr->status == '1') {
                    $this->gltransaction('deposit', $glaccttrr, $tchargeamt, null);
                    $this->create_saving_transaction_gl(null, $glaccttrr->id, null, $tchargeamt, 'debit', $request->platform, null, $this->generatetrnxref('trnxchrg'), 'reversed transfer charges', 'approved', $usern);
                }

                //reverse other charges Gl
                // if($otherglacct->status == '1'){
                //     $this->gltransaction('deposit',$otherglacct,$ocharge->amount,null);
                // $this->create_saving_transaction_gl(null,$otherglacct->id,null, $ocharge->amount,'debit',$request->platform,$request->transaction_reference,$this->generatetrnxref('otc'),'reversed others charges fees','approved',$usern);
                // }

                //reverse saving acct and current acct Gl
                if (Auth::user()->account_type == '1') { //saving acct GL

                    if ($glsavingdacct->status == '1') {
                        $this->gltransaction('withdrawal', $glsavingdacct, $totalAmount, null);
                        $this->create_saving_transaction_gl(null, $glsavingdacct->id, null, $totalAmount, 'credit', $request->platform, $request->transaction_reference, $this->generatetrnxref('D'), 'customer debited', 'approved', $usern);
                    }
                } elseif (Auth::user()->account_type == '2') { //current acct GL

                    if ($glcurrentacct->status == '1') {
                        $this->gltransaction('withdrawal', $glcurrentacct, $totalAmount, null);
                        $this->create_saving_transaction_gl(null, $glcurrentacct->id, null, $totalAmount, 'credit', $request->platform, $request->transaction_reference, $this->generatetrnxref('D'), 'customer debited', 'approved', $usern);
                    }
                }
                return response()->json(['status' => false, 'message' => $debitCustomer["message"]]);
            }

            DB::commit();

            $bank = Bank::where('bank_code', $request->bank_code)->first();

            $newdescription = empty($request->description) ? "From " . $usern : $request->description;

            $baseurl = env('APP_MODE') == "test" ? env("TEST_ACCESS_URL") : env("LIVE_ACCESS_URL");

            $url = $baseurl . "outward/funds/transfer/credit";

            $amount = (string)$request->amount;
            $bankTransfer = $this->bankTransfer($url, $this->matrixpkey, $this->matrixskey, $amount, $request->destination_account, $request->bank_code, $request->receipient_name, $usern, Auth::user()->acctno, Auth::user()->bvn, $request->transaction_reference, $newdescription);

            //return $bankTransfer;
            $this->logInfo("bank transfer response log", $bankTransfer);

            //logInfo($bankTransfer, "Monnify Transfer Response");
            $description = empty($request->description) ? "trnsf" : $request->description;
            $updtdescription = $description . "/" . $request->receipient_name . "/" . $request->destination_account . "-" . $bank->bank_name;

            $dacct = $request->receipient_name . "-" . $request->destination_account . "-" . $bank->bank_name;

            if (!empty($bankTransfer["reponse"]["data"]["response"]["ResponseCode"])) {

                if ($bankTransfer["reponse"]["data"]["response"]["ResponseCode"] == "00") {

                    $SessionID =  !empty($bankTransfer["reponse"]["data"]["response"]["SessionID"]) ? $bankTransfer["reponse"]["data"]["response"]["SessionID"] : null;

                    //companybal
                    $this->debitcreditCompanyBalance($request->amount, "debit", "combal");

                    //   $this->create_saving_transaction(null, Auth::user()->id,null,$request->amount,'debit',$request->platform,'0',$SessionID,null,null,null,
                    //   $request->transaction_reference,$updtdescription,'approved','2','trnsfer',$usern);

                    $this->updateTransactionAndAddTrnxcharges(
                        null,
                        Auth::user()->id,
                        null,
                        $charge,
                        'debit',
                        $request->platform,
                        '0',
                        $SessionID,
                        null,
                        null,
                        $request->transaction_reference,
                        $updtdescription,
                        "charges",
                        'approved',
                        '10',
                        $usern,
                        $dacct
                    );


                    $this->saveBeneficiary($request->beneficiary, $request->userid, $request->receipient_name, $request->destination_account, $request->bank_name, $request->bank_code, 'bank');

                    DB::commit();

                    $famt = " N" . number_format($request->amount, 2);
                    $dbalamt = " N" . number_format($debitCustomer['balance'], 2);
                    $bdecs1 =  $updtdescription;

                    $msg = "Debit Amt: " . $famt . "<br> Desc: " . $bdecs1 . "<br> Avail Bal: " . $dbalamt . "<br> Date:" . date('Y-m-d') . "<br> Ref: " . $request->transaction_reference;

                    $smsmsg = "Debit Amt: " . $famt . "\n Desc: " . $bdecs1 . " \n Avail Bal: " . $dbalamt . "\n Date: " . date("Y-m-d") . "\n Ref: " . $request->transaction_reference;

                    if (Auth::user()->enable_sms_alert) {
                        $this->sendSms(Auth::user()->phone, $smsmsg, $getsetvalue->getsettingskey('active_sms')); //send sms
                    }

                    if (Auth::user()->enable_email_alert) {
                        Email::create([
                            'user_id' =>  Auth::user()->id,
                            'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . ' Debit Alert',
                            'message' => $msg,
                            'recipient' => Auth::user()->email,
                        ]);

                        $this->sendMail($msg, 'Debit Transaction', Auth::user()->email, ucwords($getsetvalue->getsettingskey('company_name')) . ' Debit Alert');

                        //     Mail::send(['html' => 'mails.sendmail'],[
                        //         'msg' => $msg,
                        //             'type' => 'Debit Transaction'
                        //     ],function($mail)use($getsetvalue){
                        //         $mail->from($getsetvalue->getsettingskey('company_email'),ucwords($getsetvalue->getsettingskey('company_name')));
                        //         $mail->to(Auth::user()->email);
                        //     $mail->subject(ucwords($getsetvalue->getsettingskey('company_name')).' Debit Alert');
                        // });
                    }

                    return response()->json(['status' => true, 'message' => 'Bank Transfer Successful', 'responseCode' => '00'], 200);

                    // }elseif($bankTransfer["status"] == "99" || "63"){
                } else {

                    $SessionID =  !empty($bankTransfer["reponse"]["data"]["response"]["SessionID"]) ? $bankTransfer["reponse"]["data"]["response"]["SessionID"] : null;

                    //FAILED TRANSACTION
                    //  $this->create_saving_transaction(null, Auth::user()->id,null,$request->amount,'debit',$request->platform,'0',$SessionID,null,null,null,
                    //  $request->transaction_reference,$updtdescription,'failed','2','trnsfer',$usern);

                    $this->updateTransactionAndAddTrnxcharges(
                        null,
                        Auth::user()->id,
                        null,
                        $charge,
                        'debit',
                        $request->platform,
                        '0',
                        $SessionID,
                        null,
                        null,
                        $request->transaction_reference,
                        $updtdescription,
                        "charges",
                        'failed',
                        '10',
                        $usern,
                        $dacct
                    );


                    $this->tracktrails('1', '1', $usern, 'customer', 'Bank Transfer Failed');

                    DB::commit();

                    // $reverd = $this->ReverseDebitTrnxandcompanyGlAcct(Auth::user()->id,$totalAmount,$request->amount,$request->transaction_reference,$getsetvalue->getsettingskey('outwardoptiongl'),'bnk','Transaction reversed',$request->platform,'trnsfer',$usern,'');

                    //reverse transfer charges Gl
                    // if($glaccttrr->status == '1'){
                    //     $this->gltransaction('deposit',$glaccttrr,$tchargeamt,null);
                    // $this->create_saving_transaction_gl(null,$glaccttrr->id,null, $tchargeamt,'debit',$request->platform,null,$this->generatetrnxref('trnxchrg'),'reversed transfer charges','approved',$usern.'(c)');
                    // }

                    //reverse other charges Gl
                    // if($otherglacct->status == '1'){
                    //     $this->gltransaction('deposit',$otherglacct,$ocharge->amount,null);
                    // $this->create_saving_transaction_gl(null,$otherglacct->id,null, $ocharge->amount,'debit',$request->platform,null,$this->generatetrnxref('otc'),'reversed others charges fees','approved',$usern);
                    // }

                    //reverse saving acct and current acct Gl
                    //     if(Auth::user()->account_type == '1'){//saving acct GL

                    //     if($glsavingdacct->status == '1'){
                    // $this->gltransaction('withdrawal',$glsavingdacct,$totalAmount,null);
                    // $this->create_saving_transaction_gl(null,$glsavingdacct->id,null, $totalAmount,'credit',$request->platform,null,$this->generatetrnxref('D'),'customer debited','approved',$usern);
                    //     }

                    // }elseif(Auth::user()->account_type == '2'){//current acct GL

                    //     if($glcurrentacct->status == '1'){
                    //     $this->gltransaction('withdrawal',$glcurrentacct,$totalAmount,null);
                    // $this->create_saving_transaction_gl(null,$glcurrentacct->id,null, $totalAmount,'credit',$request->platform,null,$this->generatetrnxref('D'),'customer debited','approved',$usern);
                    //     }
                    // }


                    return response()->json(['status' => false, 'message' => 'Bank Transfer Failed', 'responseCode' => '99'], 406);


                    //    }else{

                    //         return response()->json(['status' => false, 'message' => "Transaction Completed"], 406);
                }
            } else {

                $SessionID = null;

                //FAILED TRANSACTION
                //  $this->create_saving_transaction(null, Auth::user()->id,null,$request->amount,'debit',$request->platform,'0',$SessionID,null,null,null,
                //  $request->transaction_reference,$updtdescription,'failed','2','trnsfer',$usern);

                $this->updateTransactionAndAddTrnxcharges(
                    null,
                    Auth::user()->id,
                    null,
                    $charge,
                    'debit',
                    $request->platform,
                    '0',
                    $SessionID,
                    null,
                    null,
                    $request->transaction_reference,
                    $updtdescription,
                    "charges",
                    'failed',
                    '10',
                    $usern,
                    $dacct
                );


                $this->tracktrails('1', '1', $usern, 'customer', 'Bank Transfer Failed');

                // $reverd = $this->ReverseDebitTrnxandcompanyGlAcct(Auth::user()->id,$totalAmount,$request->amount,$request->transaction_reference,$getsetvalue->getsettingskey('outwardoptiongl'),'bnk','Transaction reversed',$request->platform,'trnsfer',$usern,'');

                //reverse transfer charges Gl
                // if($glaccttrr->status == '1'){
                //     $this->gltransaction('deposit',$glaccttrr,$tchargeamt,null);
                // $this->create_saving_transaction_gl(null,$glaccttrr->id,null, $tchargeamt,'debit',$request->platform,null,$this->generatetrnxref('trnxchrg'),'reversed transfer charges','approved',$usern.'(c)');
                // }

                //reverse other charges Gl
                // if($otherglacct->status == '1'){
                //     $this->gltransaction('deposit',$otherglacct,$ocharge->amount,null);
                // $this->create_saving_transaction_gl(null,$otherglacct->id,null, $ocharge->amount,'debit',$request->platform,null,$this->generatetrnxref('otc'),'reversed others charges fees','approved',$usern);
                // }

                //reverse saving acct and current acct Gl
                //     if(Auth::user()->account_type == '1'){//saving acct GL

                //     if($glsavingdacct->status == '1'){
                // $this->gltransaction('withdrawal',$glsavingdacct,$totalAmount,null);
                // $this->create_saving_transaction_gl(null,$glsavingdacct->id,null, $totalAmount,'credit',$request->platform,null,$this->generatetrnxref('D'),'customer debited','approved',$usern);
                //     }

                // }elseif(Auth::user()->account_type == '2'){//current acct GL

                //     if($glcurrentacct->status == '1'){
                //     $this->gltransaction('withdrawal',$glcurrentacct,$totalAmount,null);
                // $this->create_saving_transaction_gl(null,$glcurrentacct->id,null, $totalAmount,'credit',$request->platform,null,$this->generatetrnxref('D'),'customer debited','approved',$usern);
                //     }
                // }

                DB::commit();

                return response()->json(['status' => false, 'message' => 'Bank Transfer Failed', 'responseCode' => '99'], 406);
            }
        } catch (LockTimeoutException $e) {

            DB::rollBack();

            return response()->json(['status' => false, 'message' => 'Error Processing Transaction'], 406);
        } finally {
            optional($lock)->release();
        }
    }

    public function transferToWalletAccount(Request $request)
    {
        $lock = Cache::lock('walltnx-' . Auth::user()->id, 3);

        try {

            $lock->block(1);

            $this->logInfo("wallet transfer", $request->except('transaction_pin'));

            $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
            $this->tracktrails('1', '1', $usern, 'customer', 'Transfer to Wallet');


            $validation = Validator::make($request->all(), [
                "amount" => "required|numeric|gt:0|max:1000000",
                "destination_account" => "required",
                "transaction_reference" => "required|string",
                "receipient_name" => "required|string",
                "transaction_pin" => "required|numeric|gt:0",
                "platform" => "required|string"
            ]);


            if ($validation->fails()) {
                $ra = array("status" => false, "message" => $validation->errors()->all()[0]);
                return response()->json($ra, 406);
            }

            DB::beginTransaction();

            $getsetvalue = new Setting();


            $description = empty($request->description) ? "trnf" : $request->description;

            $this->create_saving_transaction(
                null,
                Auth::user()->id,
                null,
                $request->amount,
                'debit',
                $request->platform,
                '0',
                null,
                null,
                null,
                null,
                $request->transaction_reference,
                $description,
                'pending',
                '2',
                'trnsfer',
                ''
            );

            $transaction = SavingsTransaction::where('reference_no', $request->transaction_reference)->where('amount', $request->amount)->first();

            if ($transaction) {
                if ($transaction->status == "approved" || $transaction->status == "failed") {
                    return response()->json(["status" => false, 'message' => "Transaction has already been completed...Please Initiate Transaction"], 409);
                }
            }

            if (Auth::user()->acctno == $request->destination_account) {
                return response()->json(['status' => false, 'message' => 'Cannot transfer to self'], 406);
            }

            $compbal = $this->validatecompanybalance($request->amount, "combal");
            if ($compbal["status"] == false) {

                $this->logInfo("validating company balance", $compbal);

                return response()->json($compbal, 406);
            }

            $chkcres = $this->checkCustomerRestriction(Auth::user()->id);
            if ($chkcres == true) {

                $this->tracktrails('1', '1', $usern, 'customer', 'Account Restricted');

                $this->logInfo("", "Customer Account Restricted");

                return response()->json(['status' => false, 'message' => 'Your Account Has Been Restricted. Please contact support'], 406);
            }

            $chklien = $this->checkCustomerLienStatus(Auth::user()->id);
            if ($chklien['status'] == true && $chklien['lien'] == 2) {


                $this->tracktrails('1', '1', $usern, 'customer', 'Account has been lien');

                $this->logInfo("Account lien", $chklien);

                return response()->json(['status' => false, 'message' => 'Your Account Has Been Lien(' . $chklien['messages'] . ')...please contact support']);
            }

            $validateuserpin = $this->validatetrnxpin($request->transaction_pin, Auth::user()->id);
            if ($validateuserpin["status"] == false) {

                $this->tracktrails('1', '1', $usern, 'customer', $validateuserpin["message"]);

                $this->logInfo("Customer pin validation", $validateuserpin);

                return response()->json($validateuserpin, 406);
            }


            $validateuserbalance = $this->validatecustomerbalance(Auth::user()->id, $request->amount);
            if ($validateuserbalance["status"] == false) {

                $this->tracktrails('1', '1', $usern, 'customer', $validateuserbalance["message"]);

                $this->logInfo("customer balance", $validateuserbalance);

                return response()->json($validateuserbalance, 406);
            }

            $validateTransferAmount = $this->validateTransfer($request->amount, $getsetvalue->getsettingskey('online_transfer'), Auth::user()->id);

            if ($validateTransferAmount['status'] == false) {

                $this->logInfo("online transfer", $validateTransferAmount);

                return response()->json($validateTransferAmount, 400);
            }

            $updtdescription = $description . "/" . $request->receipient_name . "/" . $request->destination_account;


            $debitCustomer = $this->DebitCustomerandcompanyGlAcct(Auth::user()->id, $request->amount, $request->amount, '10733842', 'w', $updtdescription, $request->platform, $request->transaction_reference, $usern);



            $this->logInfo("debit customer response", $debitCustomer);

            $dacct = $request->receipient_name . "-" . $request->destination_account;

            if ($debitCustomer["status"] == true) {

                //companybal
                $this->debitcreditCompanyBalance($request->amount, "debit", "combal");

                // $bank = banks::where('bank_code', $request->bank_code)->first();
                if ($debitCustomer["status"] == false && $debitCustomer["glstatus"] == 1) {
                    $customeracctbal = Saving::where('customer_id', Auth::user()->id)->first();
                    $customeracctbal->account_balance += $request->amount;
                    $customeracctbal->save();

                    $this->updateTransactionAndAddTrnxcharges(
                        null,
                        Auth::user()->id,
                        null,
                        0,
                        'credit',
                        $request->platform,
                        '0',
                        null,
                        null,
                        null,
                        $request->transaction_reference,
                        $updtdescription,
                        "debit reversal",
                        'approved',
                        '10',
                        $usern,
                        $dacct
                    );

                    return response()->json(["status" => false, 'message' => $debitCustomer["message"]], 400);
                }


                // $this->create_saving_transaction(null, Auth::user()->id,null,$request->amount,'debit',$request->platform,'0',null,null,null,null,
                // $request->transaction_reference,$updtdescription,'approved','2','trnsfer','');

                $this->updateTransactionAndAddTrnxcharges(
                    null,
                    Auth::user()->id,
                    null,
                    0,
                    'debit',
                    $request->platform,
                    '0',
                    null,
                    null,
                    null,
                    $request->transaction_reference,
                    $updtdescription,
                    "charges",
                    'approved',
                    '10',
                    $usern,
                    $dacct
                );


                $this->credit_account_transfer($request, $request->destination_account, '10733842', $request->transaction_reference);


                $this->saveBeneficiary($request->beneficiary, Auth::user()->id, $request->receipient_name, $request->destination_account, null, null, 'wallet');

                DB::commit();

                $msg =  "Debit Amt: N" . number_format($request->amount, 2) . "<br> Desc: " . $updtdescription . " <br>Avail Bal: N" . number_format($debitCustomer["balance"], 2) . "<br> Date: " . date("Y-m-d") . "<br>Ref: " . $request->transaction_reference;

                $smsmsg = "Debit Amt: N" . number_format($request->amount, 2) . "\n Desc: " . $updtdescription . " \n Avail Bal: N" . number_format($debitCustomer["balance"], 2) . "\n Date: " . date("Y-m-d") . "\n Ref: " . $request->transaction_reference;

                if (Auth::user()->enable_sms_alert) {
                    $this->sendSms(Auth::user()->phone, $smsmsg, $getsetvalue->getsettingskey('active_sms')); //send sms
                }

                if (Auth::user()->enable_email_alert) {

                    Email::create([
                        'user_id' => Auth::user()->id,
                        'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . ' Debit Alert',
                        'message' => $msg,
                        'recipient' => Auth::user()->email,
                    ]);

                    $this->sendMail($msg, 'Debit Transaction', Auth::user()->email, ucwords($getsetvalue->getsettingskey('company_name')) . ' Debit Alert');
                }


                return response()->json(["status" => true, 'message' => "Wallet Transfer Successful", 'responseCode' => '00', 'data' => ['balance' => $debitCustomer['balance']]]);
            } else {

                $this->updateTransactionAndAddTrnxcharges(
                    null,
                    Auth::user()->id,
                    null,
                    0,
                    'debit',
                    $request->platform,
                    '0',
                    null,
                    null,
                    null,
                    $request->transaction_reference,
                    $updtdescription,
                    "failed Transaction",
                    'failed',
                    '10',
                    '',
                    $dacct
                );

                $customeracctbal = Saving::where('customer_id', Auth::user()->id)->first();

                $newbal = $customeracctbal->account_balance + $request->amount;
                $customeracctbal->account_balance = $newbal;
                $customeracctbal->save();

                $this->create_saving_transaction(
                    null,
                    Auth::user()->id,
                    1,
                    $request->amount,
                    'credit',
                    $request->platform,
                    '0',
                    null,
                    null,
                    null,
                    null,
                    $request->transaction_reference,
                    'debit reversal',
                    'approved',
                    '4',
                    'trnsfer',
                    $usern
                );

                DB::commit();

                return response()->json(["status" => false, 'message' => "Wallet Transfer failed", 'responseCode' => '99'], 400);
            }
        } catch (LockTimeoutException $e) {

            DB::rollBack();

            return response()->json(['status' => false, 'message' => 'Error Processing Transaction'], 406);
        } finally {
            optional($lock)->release();
        }
    }

    public function credit_account_transfer($r, $acctno, $glacct, $trx)
    {

        $cust = Customer::where('acctno', $acctno)->first();
        $customeracct2 = Saving::lockForUpdate()->where('customer_id', $cust->id)->first();

        $usern = !is_null(Auth::user()->business_name) ? Auth::user()->business_name : Auth::user()->last_name . " " . Auth::user()->first_name;

        $desc =  "You recieve payment from " . $usern . " - " . $acctno;

        $getsetvalue = new Setting();

        // if($customeracct2->account_balance <= 0){
        //     $customeracct2->account_balance = $r->amount;
        //    $customeracct2->save();
        // }else{

        // }

        $newbal = $customeracct2->account_balance + $r->amount;
        $customeracct2->account_balance = $newbal;
        $customeracct2->save();

        $this->create_saving_transaction(
            null,
            $cust->id,
            null,
            $r->amount,
            'credit',
            $r->platform,
            '0',
            $r->slipno,
            null,
            null,
            null,
            $trx,
            $desc,
            'approved',
            '1',
            'trnsfer',
            ''
        );

        $glacctc = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', $glacct)->first();

        $this->tracktrails(Auth::user()->id, null, $usern, 'wallet account transfer', 'deposited to an account');

        if ($glacctc->status == '1') {
            $this->gltransaction('withdrawal', $glacctc, $r->amount, null);

            $this->create_saving_transaction_gl(Auth::user()->id, $glacctc->id, null, $r->amount, 'debit', $r->platform, null, $this->generatetrnxref('w'), $desc, 'approved', $usern);
        }

        $msg =  "Credit Amt: N" . number_format($r->amount, 2) . "<br> Desc: " . $desc . " <br>Avail Bal: N" . number_format($customeracct2->account_balance, 2) . "<br> Date: " . date("Y-m-d") . "<br>Ref: " . $trx;

        $smsmsg = "Credit Amt: N" . number_format($r->amount, 2) . "\n Desc: " . $desc . " \n Avail Bal: N" . number_format($customeracct2->account_balance, 2) . "\n Date: " . date("Y-m-d") . "\n Ref: " . $trx;

        if ($cust->enable_sms_alert) {
            $this->sendSms($cust->phone, $smsmsg, $getsetvalue->getsettingskey('active_sms')); //send sms
        }

        if ($cust->enable_email_alert) {
            Email::create([
                'user_id' => $cust->id,
                'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert',
                'message' => $msg,
                'recipient' => $cust->email,
            ]);

            $this->sendMail($msg, 'Credit Transaction', $cust->email, ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert');
        }
    }

    public function save_notification_payload(Request $r)
    {
        $this->logInfo("credit customer response", $r->all());

        NotificationPayload::create([
            'body' => json_encode($r->all())
        ]);

        $getsetvalue = new Setting();

        $customer = Customer::where('acctno', $r->accountNumber)->first();

        $branch = !is_null($customer->branch_id) ? $customer->branch_id : null;

        $trxref = $r->sessionId;

        $chekclosed = $this->checkClosedCustomer($customer->id);
        if ($chekclosed == true) {
            $this->logInfo("customer account is closed", "");

            return response()->json(["requestSuccessful" => true, "sessionId" => $trxref, "responseMessage" => "rejected transaction", "responseCode" => "02"]);
        }

        $glsavingdacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '20993097')->first();
        $glcurrentacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '20639526')->first();
        $amexpenseglacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '50792058')->first(); //am expenseGL


        if ($getsetvalue->getsettingskey('enable_virtual_ac') == '1' && $getsetvalue->getsettingskey('inwardoption') == "1") {
            $provkey = ProvidusKey::first();

            $seetlid = $r->settlementId;

            $transaction = SavingsTransaction::where('slip', $seetlid)->where('amount', $r->transactionAmount)->first();

            $authSignature = $r->header('X-Auth-Signature');

            if ($authSignature == $provkey->signature_key) {

                if ($transaction) {

                    return response()->json(["requestSuccessful" => true, "sessionId" => $trxref, "responseMessage" => "Duplicate Transaction", "responseCode" => "01"]);
                } else {

                    if (empty($customer)) {
                        return response()->json(["requestSuccessful" => true, "sessionId" => $trxref, "responseMessage" => "rejected transaction", "responseCode" => "02"]);
                    }


                    $desc = $r->tranRemarks == "" ? "inward transaction" : $r->tranRemarks;

                    $customeracct = $this->CreditCustomerAccount($customer->id, $r->settledAmount, $getsetvalue->getsettingskey('inwardoptiongl'), $trxref, $seetlid, $desc, $branch);

                    $this->checkOutstandingCustomerLoan($customer->id, $r->amount); //check if customer has an outstanding loan

                    if ($customer->account_type == '1') { //saving acct GL

                        if ($glsavingdacct->status == '1') {
                            $this->gltransaction('withdrawal', $glsavingdacct, $r->settledAmount, null);
                            $this->create_saving_transaction_gl(null, $glsavingdacct->id, $customer->branch_id, $r->settledAmount, 'credit', null, null, $this->generatetrnxref('CR'), 'customer credited', 'approved', null);
                        }
                    } elseif ($customer->account_type == '2') { //current acct GL

                        if ($glcurrentacct->status == '1') {
                            $this->gltransaction('withdrawal', $glcurrentacct, $r->settledAmount, null);
                            $this->create_saving_transaction_gl(null, $glcurrentacct->id, $customer->branch_id, $r->settledAmount, 'credit', null, null, $this->generatetrnxref('CR'), 'customer credited', 'approved', null);
                        }
                    }

                    //  if($amexpenseglacct->status == '1'){
                    //   $this->gltransaction('withdrawal',$amexpenseglacct,$r->tranxfee,null);
                    //   $this->create_saving_transaction_gl(null,$amexpenseglacct->id,null, $r->tranxfee,'debit',null,null,$this->generatetrnxref('D'),'debit asset matrix expense','approved',null);
                    //   }

                    //   $glacct = GeneralLedger::select('id','status','account_balance')->where('gl_code',$getsetvalue->getsettingskey('assetmtx'))->first();

                    //   if($glacct->status == '1'){
                    //       $this->gltransaction('deposit',$glacct,$r->tranxfee,null);
                    //         $this->create_saving_transaction_gl(null,$glacct->id,null,$r->tranxfee,'credit',null,null,$this->generatetrnxref('C'),'inward Tranx expenses','approved',null);
                    //         }

                    $msg =  "Credit Amt: N" . number_format($r->settledAmount, 2) . "<br> Desc: " . $desc . " <br>Avail Bal: N" . number_format($customeracct['balance'], 2) . "<br> Date: " . date("Y-m-d") . "<br>Ref: " . $trxref;
                    $smsmsg = "Credit Amt: N" . number_format($r->settledAmount, 2) . "\n Desc: " . $desc . " \n Avail Bal: N" . number_format($customeracct['balance'], 2) . "\n Date: " . date("Y-m-d") . "\n Ref: " . $trxref;

                    if ($customer->enable_sms_alert) {
                        $this->sendSms($customer->phone, $smsmsg, $getsetvalue->getsettingskey('active_sms')); //send sms
                    }

                    if ($customer->enable_email_alert) {
                        Email::create([
                            'user_id' => $customer->id,
                            'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert',
                            'message' => $msg,
                            'recipient' => $customer->email,
                        ]);

                        $this->sendMail($msg, 'Credit Transaction', $customer->email, ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert');


                        //  Mail::send(['html' => 'mails.sendmail'],[
                        //      'msg' => $msg,
                        //      'type' => 'Credit Transaction'
                        //  ],function($mail)use($getsetvalue,$customer){
                        //      $mail->from($getsetvalue->getsettingskey('company_email'),ucwords($getsetvalue->getsettingskey('company_name')));
                        //       $mail->to($customer->email);
                        //      $mail->subject(ucwords($getsetvalue->getsettingskey('company_name')).' Credit Alert');
                        //  });
                    }


                    return response()->json(["requestSuccessful" => true, "responseMessage" => "success", "responseCode" => "00"]);
                }
            } else {
                return response()->json(["requestSuccessful" => false, "responseMessage" => "Invalid Auth Signation Format", "responseCode" => "02"]);
            }
        } elseif ($getsetvalue->getsettingskey('enable_virtual_ac') == '1' && $getsetvalue->getsettingskey('inwardoption') == "2") {

            $desc = $r->narration == "" ? "inward transaction" : $r->narration;

            $customeracct = $this->CreditCustomerAccount($customer->id, $r->amount, $getsetvalue->getsettingskey('inwardoptiongl'), $trxref, null, $desc, $branch);

            $this->checkOutstandingCustomerLoan($customer->id, $r->amount); //check if customer has an outstanding loan

            if ($customer->account_type == '1') { //saving acct GL

                if ($glsavingdacct->status == '1') {
                    $this->gltransaction('withdrawal', $glsavingdacct, $r->amount, null);
                    $this->create_saving_transaction_gl(null, $glsavingdacct->id, $customer->branch_id, $r->amount, 'credit', null, null, $this->generatetrnxref('CR'), 'customer credited', 'approved', null);
                }
            } elseif ($customer->account_type == '2') { //current acct GL

                if ($glcurrentacct->status == '1') {
                    $this->gltransaction('withdrawal', $glcurrentacct, $r->amount, null);
                    $this->create_saving_transaction_gl(null, $glcurrentacct->id, $customer->branch_id, $r->amount, 'credit', null, null, $this->generatetrnxref('CR'), 'customer credited', 'approved', null);
                }
            }

            if ($amexpenseglacct->status == '1') {
                $this->gltransaction('withdrawal', $amexpenseglacct, $r->tranxfee, null);
                $this->create_saving_transaction_gl(null, $amexpenseglacct->id, $customer->branch_id, $r->tranxfee, 'debit', null, null, $this->generatetrnxref('D'), 'debit asset matrix expense', 'approved', null);
            }

            //substract tranxfee from GL 10897866
            $glacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '10897866')->first();

            if ($glacct->status == '1') {
                $this->gltransaction('deposit', $glacct, $r->tranxfee, null);
                $this->create_saving_transaction_gl(null, $glacct->id, $customer->branch_id, $r->tranxfee, 'credit', null, null, $this->generatetrnxref('C'), 'inward Tranx expenses', 'approved', null);
            }

            $msg =  "Credit Amt: N" . number_format($r->amount, 2) . "<br> Desc: " . $desc . " <br>Avail Bal: N" . number_format($customeracct['balance'], 2) . "<br> Date: " . date("Y-m-d") . "<br>Ref: " . $trxref;
            $smsmsg = "Credit Amt: N" . number_format($r->amount, 2) . "\n Desc: " . $desc . " \n Avail Bal: N" . number_format($customeracct['balance'], 2) . "\n Date: " . date("Y-m-d") . "\n Ref: " . $trxref;

            if ($customer->enable_sms_alert) {
                $this->sendSms($customer->phone, $smsmsg, $getsetvalue->getsettingskey('active_sms')); //send sms
            }

            if ($customer->enable_email_alert) {
                Email::create([
                    'user_id' => $customer->id,
                    'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert',
                    'message' => $msg,
                    'recipient' => $customer->email,
                ]);

                $this->sendMail($msg, 'Credit Transaction', $customer->email, ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert');

                //    Mail::send(['html' => 'mails.sendmail'],[
                //        'msg' => $msg,
                //        'type' => 'Credit Transaction'
                //    ],function($mail)use($getsetvalue,$customer){
                //        $mail->from($getsetvalue->getsettingskey('company_email'),ucwords($getsetvalue->getsettingskey('company_name')));
                //         $mail->to($customer->email);
                //        $mail->subject(ucwords($getsetvalue->getsettingskey('company_name')).' Credit Alert');
                //    });
            }
        }
    }

    public function  Bani_notification_payload(request $r)
    {
        $this->logInfo("bani credit customer response", $r->all());

        NotificationPayload::create([
            'body' => json_encode($r->all())
        ]);

        $getsetvalue = new Setting();

        $customer = Customer::where(['acctno' => $r->data["holder_account_number"], 'customerRef' => $r->data["customer_ref"]])->first();

        $branch = !is_null($customer->branch_id) ? $customer->branch_id : null;

        $trxref = $r->data["transaction_ref"];
        // $amount = $r->data["pay_amount"];
        $amount = $r->data["merch_amount"];

        $chekclosed = $this->checkClosedCustomer($customer->id);
        if ($chekclosed == true) {
            $this->logInfo("customer account is closed", "");

            return response()->json(["status" => true, "sessionId" => $trxref, "responseMessage" => "rejected transaction", "responseCode" => "02"]);
        }

        $glsavingdacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '20993097')->first();
        $glcurrentacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '20639526')->first();
        $baniglacct = GeneralLedger::select('id', 'status', 'account_balance')->where('gl_code', '10536390')->first(); //bani

        $desc = $r->data["narration"] == "" ? "inward transaction" : $r->data["narration"];

        $customeracct = $this->CreditCustomerAccount($customer->id, $amount, "10536390", $trxref, null, $desc, $branch);

        $this->checkOutstandingCustomerLoan($customer->id, $amount); //check if customer has an outstanding loan

        if ($customer->account_type == '1') { //saving acct GL

            if ($glsavingdacct->status == '1') {
                $this->gltransaction('withdrawal', $glsavingdacct, $amount, null);
                $this->create_saving_transaction_gl(null, $glsavingdacct->id, $customer->branch_id, $amount, 'credit', null, null, $this->generatetrnxref('CR'), 'customer credited', 'approved', null);
            }
        } elseif ($customer->account_type == '2') { //current acct GL

            if ($glcurrentacct->status == '1') {
                $this->gltransaction('withdrawal', $glcurrentacct, $amount, null);
                $this->create_saving_transaction_gl(null, $glcurrentacct->id, $customer->branch_id, $amount, 'credit', null, null, $this->generatetrnxref('CR'), 'customer credited', 'approved', null);
            }
        }

        // if($amexpenseglacct->status == '1'){
        //     $this->gltransaction('withdrawal',$amexpenseglacct,$r->tranxfee,null);
        //     $this->create_saving_transaction_gl(null,$amexpenseglacct->id,$customer->branch_id, $r->tranxfee,'debit',null,null,$this->generatetrnxref('D'),'debit asset matrix expense','approved',null);
        // }

        //substract tranxfee from GL 10897866
        // $glacct = GeneralLedger::select('id','status','account_balance')->where('gl_code','10897866')->first();

        // if($glacct->status == '1'){
        //     $this->gltransaction('deposit',$glacct,$r->tranxfee,null);
        //       $this->create_saving_transaction_gl(null,$glacct->id,$customer->branch_id,$r->tranxfee,'credit',null,null,$this->generatetrnxref('C'),'inward Tranx expenses','approved',null);
        //       }

        $msg =  "Credit Amt: N" . number_format($amount, 2) . "<br> Desc: " . $desc . " <br>Avail Bal: N" . number_format($customeracct['balance'], 2) . "<br> Date: " . date("Y-m-d") . "<br>Ref: " . $trxref;
        $smsmsg = "Credit Amt: N" . number_format($amount, 2) . "\n Desc: " . $desc . " \n Avail Bal: N" . number_format($customeracct['balance'], 2) . "\n Date: " . date("Y-m-d") . "\n Ref: " . $trxref;

        if ($customer->enable_sms_alert) {
            $this->sendSms($customer->phone, $smsmsg, $getsetvalue->getsettingskey('active_sms')); //send sms
        }

        if ($customer->enable_email_alert) {
            Email::create([
                'user_id' => $customer->id,
                'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert',
                'message' => $msg,
                'recipient' => $customer->email,
            ]);

            $this->sendMail($msg, 'Credit Transaction', $customer->email, ucwords($getsetvalue->getsettingskey('company_name')) . ' Credit Alert');

            //    Mail::send(['html' => 'mails.sendmail'],[
            //        'msg' => $msg,
            //        'type' => 'Credit Transaction'
            //    ],function($mail)use($getsetvalue,$customer){
            //        $mail->from($getsetvalue->getsettingskey('company_email'),ucwords($getsetvalue->getsettingskey('company_name')));
            //         $mail->to($customer->email);
            //        $mail->subject(ucwords($getsetvalue->getsettingskey('company_name')).' Credit Alert');
            //    });
        }
    }


    public function verifyWalletAccount(request $r)
    {

        $this->logInfo("verifying wallet account", $r->all());

        $validation = Validator::make($r->all(), [
            "account_number" => 'required',
        ]);
        if ($validation->fails()) {
            $ra = array("status" => false, "message" => $validation->errors()->all()[0]);
            return response()->json($ra, 406);
        }

        $cust = Customer::where('acctno', $r->account_number)->first();

        if ($cust) {

            if ($cust->id == Auth::user()->id) {
                return response()->json(['status' => false, 'message' => "Cannot Transfer to Self",]);
            }

            $ver = [
                'status' => true,
                'message' => "Account Verified Successfully",
                'data' => [
                    'first_name' => !is_null($cust->business_name) ? $cust->business_name : $cust->first_name,
                    'last_name' => !is_null($cust->business_name) ? "" : $cust->last_name
                ]
            ];

            $this->logInfo("verified", $ver);

            return response()->json($ver);
        } else {
            return response()->json(['status' => false, 'message' => "Failed to verify Wallet account",]);
        }
    }

    public function bvn_verify(Request $r)
    {

        $this->logInfo("verifying bvn", $r->all());

        $validate = Validator::make($r->all(), [
            'bvn' => ['required', 'numeric', 'digits:11'],
            'phone_number' => ['required', 'numeric', 'digits:11']
        ]);

        if ($validate->fails()) {
            return response()->json(['status' => false, 'message' =>  $validate->errors()->all()[0]]);
        }

        // $getsetvalue = new Setting();

        $token = $this->AccessMiddelware();

        $baseurl = env('APP_MODE') == "test" ? env("TEST_ACCESS_URL") : env("LIVE_ACCESS_URL");

        $response = Http::withHeaders([
            "Content-Type" => "application/json",
            "Authorization" => "Bearer " . $token['token'],
        ])->post($baseurl . "outward/bvn/verify", [
            "bvn" => $r->bvn,
            "phone_number" => $r->phone_number
        ])->json();

        $this->logInfo("verifying bvn response", $response);

        if ($response['data']['response_code'] == "00") {
            return response()->json(["status" => true, "message" => $response['data']['message'], "data" => $response['data']], 200);
        } else {
            $msg = !empty($response['data']['message']) ? $response['data']['message'] : "Invalid bvn";
            return response()->json(["status" => false, "message" => $msg], 406);
        }
    }

    public function validateMatixpayBalance($accountno, $amout)
    {
        $response = [];

        $this->logInfo("check matrxpay balance Url", $this->matrixurl . "get-balance/--" . $accountno);

        $checkbalanace = Http::withHeaders([
            "publickey" => $this->matrixpkey,
            "secretkey" => $this->matrixskey,
            "content_Type" => "application/json"
        ])->post($this->matrixurl . "get-balance", [
            "accountNumber" => $accountno
        ])->json();

        $this->logInfo("validating matrixpay balance", $checkbalanace);
        //return $checkbalanace;

        if ($checkbalanace["balance"] < $amout) {
            $response = ["status" => false, 'message' => "Switcher OR Inoperative Error... Please contact support"];
        } else {
            $response = ["status" => true, 'message' => "Amount is Valid",];
        }
        return $response;
    }


    public function transaction_recipt($ref)
    {

        $query = SavingsTransaction::select('amount', 'type', 'reference_no', 'notes', 'status', 'trnx_type', 'created_at')->where('reference_no', $ref)->first();

        if ($query) {

            $data = [
                'reference_no' => $query->reference_no,
                'created_at' => $query->created_at,
                'type' => $query->type,
                'status' => $query->status,
                'amount' => $query->amount,
                'notes' => $query->notes
            ];

            return response()->json(['status' => true, 'message' => 'reciept found', 'data' => $data]);
        } else {

            return response()->json(['status' => true, 'message' => 'reciept not found', 'data' => []]);
        }
    }
}
