<?php

namespace App\Http\Controllers\Api;

use App\Models\Email;
use App\Services\MatrixPayService;
use App\Models\Setting;
use App\Jobs\SendSmsJob;
use App\Models\Customer;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Traits\HttpResponses;
use App\Models\SavingsProduct;
use Illuminate\Support\Carbon;
use App\Http\Traites\UserTraite;
use App\Http\Traites\AuditTraite;
use App\Models\PhoneVerification;
use App\Http\Traites\SavingTraite;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Cache;
use App\Notifications\AppNotification;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Notification;
use Illuminate\Contracts\Cache\LockTimeoutException;

class RegisterController extends Controller
{
    use SavingTraite;
    use AuditTraite;
    use UserTraite;

    use HttpResponses;


    public function __construct(
        private MatrixPayService $matrixPay,
    ) {}


    public function sendPhoneOtp(Request $r)
    {

        $clientIp = $r->header('ipaddr') ?? $r->ip();
        $dvtyp = ($r->deviceType === 'web') ? "web- {$clientIp}" : "mobile- {$clientIp}";
        $this->logInfo($r->all(), "Phone OTP request {$r->getClientIp()} {$dvtyp}");

        $validation = Validator::make($r->all(), [
            'phone' => ['required', 'numeric', 'digits:11'],
            'device_id' => ['required'],

        ]);

        if ($validation->fails()) {
            return $this->error(message: 'Validation error.', responseCode: 422, errors: $validation->errors()->toArray());
        }

        $device = $r->header('device_id') ?? $r->device_id;
        if (empty($device)) {
            return $this->error(message: 'Invalid device ID.');
        }

        $customer = PhoneVerification::where('phone', $r->phone)->first();

        if ($customer) {
            return $this->error(message: 'Customer already exists with this phone number. Please use the login option.', responseCode: '201');
        }

        $otpCode = $this->generateSixOTP();
        $expiry = Carbon::now()->addMinutes(10)->format('Y-m-d H:i:s');

        PhoneVerification::create([
            'phone' => $r->phone,
            'otp' => $otpCode,
            'otp_expired_at' => $expiry,
            'is_verify' => 0,
            'ip_address' => $r->getClientIp(),
            'device_id' => $device,
        ]);

        $msg = $otpCode . " is your AssetMatrix MFB One Time Password";

        Email::create([
            'uuid' => Str::uuid(),
            'user_id' => null,
            'subject' => 'phone verification',
            'message' => $msg,
            'recipient' => $r->phone,
            'branch_id' => '1'
        ]);

        SendSmsJob::dispatch($r->phone, $msg);

        return $this->success(
            message: 'OTP sent successfully.'
        );
    }

    public function verifyPhoneOtp(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'phone' => ['required', 'numeric', 'digits:11'],
            'otp'   => ['required', 'numeric', 'digits:6'],
        ]);

        if ($validation->fails()) {
            return $this->error(message: 'Validation error.', responseCode: 422, errors: $validation->errors()->toArray());
        }

        $record = PhoneVerification::where('phone', $request->phone)
            ->where('otp', $request->otp)
            ->first();

        if (!$record) {
            return $this->error(message: 'Invalid OTP or phone number.', responseCode: 400);
        }

        if (Carbon::now()->greaterThan(Carbon::parse($record->otp_expired_at))) {
            return $this->error(message: 'OTP has expired. Please request a new one.', responseCode: 410);
        }

        $record->is_verify = 1;
        $record->save();

        return $this->success(message: 'Phone number verified successfully.');
    }

    public function resendPhoneOtp(Request $r)
    {
        $MAX_RESENDS = 3;
        $WINDOW_MINUTES = 120;
        $COOLDOWN_SECONDS = 120;

        $validation = Validator::make($r->all(), [
            'phone' => ['required', 'numeric', 'digits:11'],
            'device_id' => ['required']
        ]);

        if ($validation->fails()) {
            return $this->error('Validation error.', 422, $validation->errors()->toArray());
        }

        $phone = $r->phone;
        $device = $r->header('device_id') ?? $r->device_id;
        if (empty($device)) {
            return $this->error('Invalid device ID.', 400);
        }
        $cooldownKey = "otp_cooldown_{$phone}";
        if (Cache::has($cooldownKey)) {
            $secondsLeft = Cache::get($cooldownKey) - time();
            $secondsLeft = $secondsLeft > 0 ? $secondsLeft : 0;
            return $this->error("You are requesting OTPs too quickly. Try again in {$secondsLeft} seconds.", responseCode: 429);
        }

        $countKey = "otp_resend_count_{$phone}";
        $resendCount = Cache::get($countKey, 0);
        if ($resendCount >= $MAX_RESENDS) {
            return $this->error(message: "You have exceeded the maximum number of OTP resend attempts. Try again later.", responseCode: 429);
        }

        $record = PhoneVerification::where('phone', $phone)->first();

        if ($record && $record->is_verify) {
            return $this->error('Phone number is already verified.', 409);
        }

        $otpCode = $this->generateSixOTP();
        $expiry = Carbon::now()->addMinutes(10)->format('Y-m-d H:i:s');

        if ($record) {
            $record->update([
                'otp' => $otpCode,
                'otp_expired_at' => $expiry,
                'is_verify' => 0,
                'ip_address' => $r->getClientIp(),
                'device_id' => $device,
            ]);
        } else {
            PhoneVerification::create([
                'phone' => $phone,
                'otp' => $otpCode,
                'otp_expired_at' => $expiry,
                'is_verify' => 0,
                'ip_address' => $r->getClientIp(),
                'device_id' => $device,
            ]);
        }

        $msg = $otpCode . " is your AssetMatrix MFB One Time Password";

        Email::create([
            'uuid' => Str::uuid(),
            'user_id' => null,
            'subject' => 'phone verification - resend',
            'message' => $msg,
            'recipient' => $phone,
            'branch_id' => '1'
        ]);

        SendSmsJob::dispatch($phone, $msg);

        Cache::put($cooldownKey, time() + $COOLDOWN_SECONDS, $COOLDOWN_SECONDS);
        Cache::put($countKey, $resendCount + 1, now()->addMinutes($WINDOW_MINUTES));

        return $this->success(message: 'OTP resent successfully.');
    }


    public function register(Request $request)
    {

        $lock = Cache::lock('register_user', 10);
        try {
            $lock->block(5);
            $this->logInfo("Account registration log", $request->all());
            $validator = Validator::make($request->all(), [
                'first_name' => 'required|string|max:50',
                'last_name' => 'required|string|max:50',
                'phone' => 'required|string|numeric|digits:11|unique:customers,phone',
                'dob' => 'required|string',
                'gender' => 'required|string',
                'username' => 'required|string|unique:customers,username',
                'pin' => 'required|numeric|digits:4',
                'password' => 'required|string|min:8',
                'email' => 'required|string|min:8',
                'bvn' => 'required|string|unique:customers,bvn',
                'device_id' => 'nullable|string',
                'device_token' => 'nullable|string',
            ]);


            if ($validator->fails()) {
                return $this->error(
                    message: 'Validation error.',
                    responseCode: 422,
                    errors: $validator->errors()->toArray()
                );
            }



            $verifyPhone = PhoneVerification::where('phone', $request->phone)->first();
            if (!$verifyPhone) {
                return $this->error(message: "Phone number does not exists", responseCode: 409);
            }

            if ($verifyPhone->is_verify !== 1) {
                return $this->error("Phone number has not been verified", 409);
            }

            $otpCode = $this->generateSixOTP();
            $getsetvalue = new Setting();

            if (!empty($request->email)) {
                if (Customer::where('email', $request->email)->exists()) {
                    return $this->error("Account with this email already exists", 409);
                }
            }

            if (Customer::where('phone', $request->phone)->exists()) {
                return $this->error("Account with this phone number already exists", 409);
            }

            if (Customer::where('username', $request->username)->exists()) {
                return $this->error("Account with this username already exists", 409);
            }

            if (!empty($request->bvn)) {
                if (strlen($request->bvn) != 11) {
                    return $this->error('Invalid BVN length', 406);
                }
                if (!is_numeric($request->bvn)) {
                    return $this->error('BVN must be numeric', 406);
                }
                if (Customer::where('bvn', $request->bvn)->exists()) {
                    return $this->error('Account with this BVN already exists', 409);
                }
            }

            $refe = Str::random(6);
            $account_number = $this->generateUniqueAccountNumber();

            $saviprod = SavingsProduct::select('id')->get();
            $acctCategory = !empty($request->accttype) && $request->accttype == "cooperate"
                ? $saviprod[1]["id"]
                : $saviprod[0]["id"];

            $userdata = [
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'phone' => $request->phone,
                'gender' => strtolower($request->gender),
                'dob' => date('Y-m-d', strtotime($request->dob)),
                'account_type' => $request->accttype == "cooperate" ? '2' : '1',
                'bvn' => $request->bvn,
                'account_category' => $acctCategory,
                'acctno' => $account_number,
                'username' => $request->username,
                'password' => Hash::make($request->password),
                'pin' => Hash::make($request->pin),
                'otp' => $otpCode,
                'otp_expiration_date' => Carbon::now()->addMinutes(10),
                'referral_code' => strtolower($refe),
                'transfer_limit' => '50000',
                'online_transfer_limit' => '50000',
                'reg_date' => Carbon::now(),
                'source' => 'online',
                'status' => $request->accttype == "cooperate" ? '7' : '1',
                'enable_email_alert' => '1',
                'enable_sms_alert' => '1',
                'business_name' => $request->business_name,
                'ctype' => $request->accttype == "cooperate" ? "cooperate" : "savings",
                'referee_name' => $request->ref_name,
                'referee_phone' => $request->ref_phone,
                'referee_bank' => $request->ref_bank,
                'referee_account_no' => $request->ref_account_number,
                'refferre_name' => $request->reff_name,
                'refferre_phone' => $request->reff_phone,
                'refferre_bank' => $request->reff_bank,
                'refferre_account_no' => $request->reff_account_number,
                'cac_no' => $request->cac_no,
                'incorporated_date' => $request->incorporatde_date,
                'tax_id' => $request->tax_id,
                'device_id' => $request->device_id,
                'device_token' => $request->device_token,
                'phone_verify' => 1
            ];

            // We can't integrate MatrixPay account creation at the moment


            $customer = Customer::create($userdata);

            // Create Savigs Records
            $this->create_account(null, $customer->id, '1');
            
            // Create MatrixPay Virtual Account
              $this->matrixPay->createAccount($customer);


            $name = $request->accttype == "cooperate"
                ? ucwords($request->business_name)
                : ucwords($request->last_name);

            $account_name = $request->first_name . " " . $request->last_name;

            // $msg = "Welcome {$name}, below is your account details:<br>
            //     Username: {$request->username}<br>
            //     Account No: {$account_number}<br>
            //     Bank: " . ucwords($getsetvalue->getsettingskey('company_name')) . "<br><br>";

            $msg = "
                <p>Dear {$name},</p>

                <p>Your account has been successfully created. Below are your account details:</p>

                <p>
                <strong>Account Name:</strong> {$account_name}<br>
                <strong>Account Number:</strong> {$account_number}<br>
                 <strong>Username:</strong> {$request->username}<br>
                <strong>Bank:</strong> " . ucwords($getsetvalue->getsettingskey('company_name')) . "
                </p>
                <p>
                // Thank you for choosing " . ucwords($getsetvalue->getsettingskey('company_name')) . ".
                // </p>
                <p>
                Warm regards,<br>
                <strong>" . ucwords($getsetvalue->getsettingskey('company_name')) . " Team</strong>
                </p>
                ";

            Email::create([
                'user_id' => $customer->id,
                'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . " Account Registration",
                'message' => $msg,
                'recipient' => $request->email,
            ]);

            Notification::route('mail', $request->email)
                ->notify(new AppNotification('Registration Successful', $msg));

            // SendSmsJob::dispatch($request->phone, $msg);


            $fullName = $request->last_name . " " . $request->first_name;
            $this->tracktrails('1', '1', $fullName, 'customer', 'new customer account created');

            $this->logInfo("Account Registration Successful", $userdata);

            return $this->success('Account Created Successfully', [
                'customer_id' => $customer->id,
                'account_number' => $account_number,
                'username' => $request->username,
            ]);
        } catch (LockTimeoutException $e) {

            return $this->error('Error Processing Registration', 406);
        } finally {
            optional($lock)->release();
        }
    }

    public function existingAccount(Request $r)
    {
        $this->logInfo("Verifying Exiting Account", $r->all());

        $validation = Validator::make($r->all(), [
            "account_number" => 'required|string|max:10',
        ]);

        if ($validation->fails()) {
            $ra = array("status" => false, "message" => $validation->errors()->all()[0]);
            return response()->json($ra, 406);
        }
        $otpCode = $this->generateSixOTP();

        $getsetvalue = new Setting();

        $exacct = Customer::where('acctno', $r->account_number)->first();

        $passwrd = mt_rand("11111111", "99999999");

        if ($exacct) {
            if ($exacct->phone_verify == 0) {
                $exacct->username = $r->account_number;
                $exacct->password = Hash::make($passwrd);
                $exacct->status = '1';
                $exacct->enable_email_alert = '1';
                $exacct->enable_sms_alert = '1';
                $exacct->phone_verify = '1';
                $exacct->save();

                $ver =   ['status' => true, 'acctexist' => 1, 'email' => $exacct->email, 'message' => "Account Verified Successfully"];

                $this->logInfo("verified", $ver);


                $msg = "welcome " . $exacct->last_name . " " . $exacct->first_name . " <br>Below is your Login Details <br> Username: " . $r->account_number . " Password: " . $passwrd;

                if (!is_null($exacct->email)) {

                    Email::create([
                        'user_id' => $exacct->id,
                        'subject' => ucwords($getsetvalue->getsettingskey('company_name')) . " Account Confirmation",
                        'message' => $msg,
                        'recipient' => $exacct->email,
                    ]);

                    Notification::route('mail', $exacct->email)
                        ->notify(new AppNotification('Account Confirmation', $msg));
                }

                return response()->json($ver);
            } else {
                return response()->json(['status' => false, 'message' => "Account already linked...Please Login",]);
            }
        } else {
            return response()->json(['status' => false, 'message' => "Account Number Not Found",]);
        }
    }
}
