<?php

namespace App\Exceptions;

use App\Traits\HttpResponses;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Http\Exceptions\ThrottleRequestsException;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpKernel\Exception\HttpExceptionInterface;
use Symfony\Component\HttpKernel\Exception\MethodNotAllowedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException;
use Throwable;

class Handler extends ExceptionHandler
{
    use HttpResponses;

    /**
     * A list of exception types with their corresponding custom log levels.
     *
     * @var array<class-string<\Throwable>, \Psr\Log\LogLevel::*>
     */
    protected $levels = [];

    /**
     * A list of the exception types that are not reported.
     *
     * @var array<int, class-string<\Throwable>>
     */
    protected $dontReport = [];

    /**
     * A list of the inputs that are never flashed to the session on validation exceptions.
     *
     * @var array<int, string>
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    public function register(): void
    {
        // $this->reportable(function (Throwable $e) {});
    }


public function render($request, Throwable $e)
{
    if (! $request->expectsJson() && ! $request->is('api/*')) {
        return parent::render($request, $e);
    }

    $http = 500;
    $code = 500;
    $message = 'Internal server error.';
    $errors = [];

    if ($e instanceof ValidationException) {
        $http = $code = 422;
        $errors = $e->errors();
        $message = 'Validation failed.';
    } elseif ($e instanceof AuthenticationException) {
        $http = $code = 401;
        $message = 'Unauthenticated.';
    } elseif ($e instanceof AuthorizationException) {
        $http = $code = 403;
        $message = $e->getMessage() ?: 'Forbidden.';
    } elseif ($e instanceof ThrottleRequestsException || $e instanceof TooManyRequestsHttpException) {
        $http = $code = 429;
        $message = 'Too many requests.';
    } elseif ($e instanceof ModelNotFoundException) {
        $http = $code = 404;
        $model = class_basename($e->getModel() ?? 'Resource');
        $message = "{$model} not found.";
    } elseif ($e instanceof NotFoundHttpException) {
        $http = $code = 404;
        $message = 'Route not found.';
    } elseif ($e instanceof MethodNotAllowedHttpException) {
        $http = $code = 405;
        $message = 'Method not allowed.';
    } elseif ($e instanceof HttpExceptionInterface) {
        $http = $code = $e->getStatusCode();
        $message = $e->getMessage() ?: 'HTTP error.';
    } else {
        $http = $code = 500;
        $message = 'Internal server error.';
    }

    return $this->error(
        message: $message,
        errors: $errors
    )->setStatusCode($http);
}

}
